# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Test code using HTTP requests.
"""


import time
import unittest
import subprocess
import os
import urllib2

import pkg_resources

from getmediumurl import Matcher
from getmediumurl.cache import cachedproperty
from getmediumurl.reader import NotFoundError
from getmediumurl.urlreaders import DEFAULT_URLREADER
from getmediumurl.htmlplugin import HTMLPlugin, HTMLMedium
from getmediumurl.xmlhtml import read_xml

from tests_getmediumurl.shared import requirehtml


__all__ = ("HTTPInteractionTestCase",)


class MyMedium(HTMLMedium):

    """A subclass of `HTMLMedium` for tests."""

    @cachedproperty
    def raw_page_url(self):
        """Return the matched URL."""
        return self.mediumid.swapcase()


class SimplePlugin(HTMLPlugin):

    """A simple plugin to test getting titles.

    The URL with case swapped is stored as `mediumid`, to make using
    different ports simple and to detect code using `mediumid` instead
    of `get_url`.
    """

    medium_class = MyMedium

    @cachedproperty
    def raw_page_url(self):
        """Return the matched URL."""
        return self.mediumid.swapcase()

    @classmethod
    def match(cls, url, urlreader):
        """Match any URL, store it as `mediumid`."""
        return cls(url.swapcase(), urlreader)


class HTTPInteractionTestCase(unittest.TestCase):

    """Tests."""

    def setUp(self):
        """Prepare."""
        self.matcher = Matcher(SimplePlugin)
        # Start an HTTP server.
        port = 8000
        self.host = "http://localhost:%d/" % port
        htdocs = pkg_resources.resource_filename("tests_getmediumurl",
                                                 "htdocs")
        self.httpd = subprocess.Popen(
            ("python2", "-m", "SimpleHTTPServer", str(port)),
            cwd=htdocs,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE)
        # Wait until the server is ready.
        while True:
            try:
                urllib2.urlopen(self.host)
            except (urllib2.URLError, ValueError):
                time.sleep(0.1)
            else:
                break

    def tearDown(self):
        """Stop the server."""
        try:
            self.httpd.terminate()
        except AttributeError:  # Python 2.5
            import signal
            os.kill(self.httpd.pid, signal.SIGTERM)
        self.httpd.wait()

    @requirehtml
    def test_not_disabled(self):
        """Test that testing plugin is not disabled"""
        for msg in SimplePlugin.disabled():
            self.assertEqual(msg, None)

    @requirehtml
    def test_get_title(self):
        """Test default get_title"""
        plugin = self.matcher.match(self.host + "titled.html")
        self.assertEquals(" ".join(plugin.get_medium().title.split()),
                          "A Very Complex Title Full of Simple Words")

    def test_get_xml_title(self):
        """Test get_xml_title"""
        title = read_xml(DEFAULT_URLREADER(self.host + "titled.xml").content)
        self.assertEquals(" ".join(title.findtext("//title").split()),
                          "Another Very Complex Title Full of Simple Words")

    @requirehtml
    def test_canonical(self):
        """Test determining canonical URLs"""
        for uncanonical, canonical in (
            ("uncanonical.html", self.host + "canonical.html"),
            ("uncanonical-2.html", self.host + "canonical.html"),
            ("uncanonical-3.html", "http://example.com/canonical.html"),
            ("uncanonical-4.html", "http://example.org/canonical.html"),
            ):
            plugin = self.matcher.match(self.host + uncanonical)
            self.assertEquals(plugin.get_medium().url, canonical)

    @requirehtml
    def test_unicode_title(self):
        """Test obtaining non-ASCII title"""
        plugin = self.matcher.match(self.host + "unicode.html")
        self.assertEqual(plugin.get_medium().title,
                         u"A document of ﬆrange title")

    def test_notfound(self):
        """Test that non-existent documents won't be found"""
        self.assertRaises(NotFoundError, DEFAULT_URLREADER,
                          self.host + "nil.html")

    def test_twice(self):
        """Test that a single URL can be read twice."""
        first = DEFAULT_URLREADER(self.host + "unicode.html").content
        second = DEFAULT_URLREADER("%s%s" % (self.host,
                                             "unicode.html")).content
        self.assertEqual(first, second)
