# -*- coding: utf-8 -*-
# Copyright (C) 2010, 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Finding plugins and matching URLs to them.
"""


from urlreader import URLReader

from getmediumurl.compat import OrderedDict
from getmediumurl.plugin import Plugin
import getmediumurl.plugins
from getmediumurl.utils import make_sequence, deprecated


class Matcher(object):

    """An object matching plugins to media.

    :param plugins: an object specifying which plugins may be used, it
       is passed to the `add_plugins` method
    :param urlreader: initial value of the `urlreader` attribute

    .. warning::

       Some plugins may access any URL or local file and fail on
       specially prepared input.  Specify an explicit list of plugins
       if using it on untrusted media URLs.

    """

    def __init__(self, plugins=None, urlreader=None):
        """Make a new matcher."""
        #: A `collections.OrderedDict` mapping plugin class names
        #: (without their module names) to their classes.
        #:
        #: If the Python version used does not have the
        #: `collections.OrderedDict` class, then an internal class
        #: implementing a subset of its methods is used.
        self.plugins = OrderedDict()
        #: An `OrderedDict` of non-usable plugins.
        #:
        #: Keys are plugin class names, values are lists of strings
        #: describing why the plugins cannot be used.
        self.disabled = OrderedDict()
        self.add_plugins(plugins)
        #: An instance of `urlreader.urlreader.URLReader` used by
        #: plugins instantiated by this matcher to obtain documents
        #: from URLs.
        if urlreader is None:
            urlreader = URLReader()
        self.urlreader = urlreader

    def add_plugins(self, plugins):
        """Add more plugins to the matcher.

        The *plugins* argument is an object specifying which plugins
        will be added, it is one of the following:

        * a module of which all classes derived from
          `getmediumurl.plugin.Plugin` will be used as plugins

        * a string naming such module

        * a class derived from `getmediumurl.plugin.Plugin`

        * a string naming such class

        * an iterable of above ones

        * `None`, replaced with all plugins included with GetMediumURL
          (from modules in the `getmediumurl.plugins` package)

        Only classes deriving from `getmediumurl.plugin.Plugin` whose
        `getmediumurl.plugin.Plugin.disabled` method yields no element
        will be used.
        """
        if plugins is None:
            plugins = ("getmediumurl.plugins.%s" % module
                       for module in getmediumurl.plugins.__all__)
        # Replace single plugin by a tuple of it.
        else:
            plugins = make_sequence(plugins)
        for plugin in plugins:
            if self._add_plugin(plugin):
                continue
            if isinstance(plugin, basestring):
                submodules = plugin.split(".")
                try:
                    module = __import__(plugin)
                except ImportError:
                    continue
                for attr in submodules[1:]:
                    module = getattr(module, attr)
            else:
                module = plugin
            if hasattr(module, "__all__"):
                attributes = module.__all__
            else:
                attributes = dir(module)
            for attr in attributes:
                klass = getattr(module, attr)
                self._add_plugin(klass)

    def _add_plugin(self, plugin):
        """Add a class *plugin* to `plugins` if valid.

        Returns `False` if *plugin* is not a class.
        """
        try:
            if not issubclass(plugin, Plugin):
                return True
        except TypeError:
            return False
        reasons = list(plugin.disabled())
        if reasons:
            self.disabled[plugin.__name__] = reasons
        else:
            self.plugins[plugin.__name__] = plugin

    def __repr__(self):
        """Return string representation of the plugin matcher."""
        plugins = self.plugins.values()
        return "<%s %s>" % (self.__class__.__name__, repr(plugins))

    def match(self, url, fast=False):
        """Match an URL to a known plugin.

        Returns a `getmediumurl.plugin.Plugin` object or `None` if no
        plugin matched this URL.

        :param url: the URL of the video page to match, the one
          typically seen by users

        :param fast: set it to `True` to not match plugins which
          support many sites but match slowly (this option is designed
          to be used when matching many URLs of which only some will
          be matched)
        """
        iterable = self.plugins.itervalues()
        for plugin in iterable:
            if fast and not plugin.fast:
                continue
            obj = plugin.match(url, self)
            if obj is not None:
                return obj

    def get_plugin(self, name, mediumid):
        """
        .. deprecated:: 0.0a3
           Don't use it, will be removed in 0.0a4.

        Get plugin of specific class *name* and *mediumid*.

        Returns a `getmediumurl.plugin.Plugin` object or `None` if no
        known plugin has this name.

        This can be used to get more data about media in a different
        process, but it should be more efficient to store all the data
        obtained once.
        """
        deprecated("don't use this function")
        plugin = self.plugins.get(name, None)
        if plugin is not None:
            return plugin(mediumid, self)
