# -*- coding: utf-8 -*-
# Copyright (C) 2010, 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Information about a single medium on a page with possible more.
"""


from urlreader.propcache import cachedproperty


__all__ = ("Medium",)


class Medium(object):

    """
    Base class for objects representing a single medium (one of media)
    on a page.

    This class should not be directly instantiated and its subclasses
    should be instantiated only inside `getmediumurl.plugin.Plugin` or
    its subclasses.

    Instances of this class can be iterated to get instances of
    `getmediumurl.format.Format` representing formats in which a file
    of this medium can be downloaded.
    """

    #: A string where ``%s`` will be replace by `mediumid` to make the
    #: URL matched by `match_re`.
    url_format = None

    #: A string returned by default `website` if not `None`.
    #:
    #: By default the class name is used.
    website_name = None

    def __init__(self, plugin):
        """Make a medium object."""
        #: A string of plugin-specific format, with the plugin class
        #: it can uniquely determine medium.
        self.mediumid = plugin.mediumid
        #: A subclass of `getmediumurl.reader.URLReader`.
        self.urlreader = plugin.urlreader

    @cachedproperty
    def website(self):
        """Name of the website containing medium.

        Unless overridden by a subclass, it is the value of
        `website_name` attribute.
        """
        assert self.website_name is not None
        return self.website_name

    @cachedproperty
    def title(self):
        """Medium's title, or `None` if untitled."""
        assert self
        return None

    @cachedproperty
    def url(self):
        """URL which was matched to the plugin, or equivalent.

        Default implementation uses the `url_format` class attribute.
        """
        return self.url_format % self.mediumid

    def __iter__(self):
        """Return an iterable of medium formats.

        Each element yielded is an instance of
        `getmediumurl.format.Format` and represents a format in which
        a medium file can be downloaded.

        Default implementation returns an empty sequence.
        """
        assert self
        return iter(())

    @cachedproperty
    def thumbnail(self):
        """URL of a thumbnail image or `None` if not found."""
        assert self
        return None

    @cachedproperty
    def description(self):
        """Medium description or `None` if unknown."""
        assert self
        return None

    @cachedproperty
    def license(self):
        """Medium license URL or `None` if unknown."""
        assert self
        return None

    @cachedproperty
    def language(self):
        """Language code, as described in :rfc:`5646`.

        An empty string is used by default, plugins which can
        determine the language should override this property.
        """
        assert self
        return u""

    @cachedproperty
    def author_name(self):
        """Name or user identification of medium's author."""
        assert self
        return None

    @cachedproperty
    def author_url(self):
        """URL of medium's author."""
        assert self
        return None
