# -*- coding: utf-8 -*-
# Copyright (C) 2010, 2012  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Code for compatibility with old versions of Python.
"""


try:

    from collections import OrderedDict
    assert OrderedDict

except ImportError:

    class OrderedDict(object):

        """An ordered dictionary-like class implemented using a list.

        It provides a subset of `collections.OrderedDict`
        functionality used by other modules of GetMediumURL.

        When using Python 2.7, 3.1 or newer, `collections.OrderedDict`
        is imported instead of this class.
        """

        def __init__(self):
            """Make an empty `OrderedDict`."""
            self.elements = []

        def __setitem__(self, key, value):
            """Append mapping of `key` to `value`."""
            if not key in self:
                self.elements.append((key, value))

        def iterkeys(self):
            """Iterate over keys."""
            for element in self.elements:
                yield element[0]

        def itervalues(self):
            """Iterate over values."""
            for element in self.elements:
                yield element[1]

        def iteritems(self):
            """Iterate over ``(key, value)`` pairs."""
            for element in self.elements:
                yield element

        def values(self):
            """Return a list of values."""
            return list(self.itervalues())

        def get(self, key, default):
            """Return value of given `key`, or `default` if not found."""
            for element_key, value in self.elements:
                if key == element_key:
                    return value
            return default

        def __contains__(self, key):
            """Check if the given `key` is found."""
            for element in self.elements:
                if key == element[0]:
                    return True
            return False

        def __nonzero__(self):
            """Check if the list is non-empty."""
            return bool(self.elements)


try:
    from io import BytesIO as StringIO
    assert StringIO
except ImportError:
    try:
        from cStringIO import StringIO
        assert StringIO
    except ImportError:
        from io import StringIO
        assert StringIO


import sys


if sys.version_info[0] == 2:
    IS_PY2K = True
    basestring = basestring
    unicode = unicode
    xrange = xrange
    from urllib import unquote
    from urllib2 import urlopen, URLError
    from urlparse import urljoin, urlparse, parse_qs
    assert basestring and unicode and xrange and unquote and urlopen \
        and URLError and urljoin and urlparse and parse_qs
else:
    IS_PY2K = False
    basestring = str
    unicode = str
    xrange = range
    from urllib.parse import urljoin, urlparse, parse_qs, unquote
    from urllib.request import urlopen
    from urllib.error import URLError
    assert basestring and unicode and xrange and unquote and urlopen \
        and URLError and urljoin and urlparse and parse_qs


__all__ = ("OrderedDict", "StringIO", "basestring", "urljoin", "unicode",
           "urlparse", "parse_qs", "unquote", "urlopen", "URLError")
