# -*- coding: utf-8 -*-
# Copyright (C) 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Plugin for some sites using SWFObject."""


from __future__ import unicode_literals

import re

from urlreader.propcache import cachedproperty

from getmediumurl.compat import urlparse, unquote
from getmediumurl.htmlplugin import HTMLPlugin, HTMLMedium
from getmediumurl.format import Format


__all__ = ("SWFObject",)


#: Regular expression for addVariable calls.  It's obviously incorrect
#: and could work in some cases.
_ADDVAR = re.compile(r"addVariable\((?P<name>(?:'[^']+'|\"[^\"]+\")),"
                     r"\s*(?P<value>(?:'[^']+'|\"[^\"]+\"))\);")


class SWFObjectMedium(HTMLMedium):

    """A video using SWFObject."""

    url_format = "%s"

    @cachedproperty
    def _params(self):
        """A dictionary of SWFObject variables."""
        params = {}
        for element in self.page.getiterator("script"):
            if not element.text:
                continue
            for match in _ADDVAR.finditer(element.text):
                value = match.group("value")[1:-1]
                if value.startswith("http%3A%2F%2F"):
                    value = unquote(value)
                params[match.group("name")[1:-1]] = value
        return params

    def __iter__(self):
        """Yield formats."""
        yield Format(url=self._params["file"])

    @cachedproperty
    def thumbnail(self):
        """Thumbnail URL."""
        for key in "image", "thumbnail":
            try:
                return self._params[key]
            except KeyError:
                continue
        return super(SWFObjectMedium, self).thumbnail

    @cachedproperty
    def website(self):
        """Website name."""
        name = super(SWFObjectMedium, self).website
        if name != "SWFObjectMedium":
            return name
        return urlparse(self.url).hostname


class SWFObject(HTMLPlugin):

    """Plugin for sites using SWFObject."""

    url_format = "%s"
    medium_class = SWFObjectMedium

    @classmethod
    def match(cls, url, matcher):
        """Match URL to the plugin."""
        try:
            document = matcher.urlreader(url)
        except ValueError:
            return None
        if not document.content_type.startswith("text/html"):
            return None
        if document.content.find("SWFObject") == -1:
            return None
        instance = cls(url, matcher)
        for medium in instance:
            for fmt in medium:
                assert fmt
                return instance
