# -*- coding: utf-8 -*-

import sys
import traceback

from . import log
from .server import BaseHTTPRequestHandler
from .utils import force_bytes
from .templating import render_file
from .locales import localeman

class GitLayerRequestHandler(BaseHTTPRequestHandler):
    '''
        BaseHTTPRequestHandler extension for HTTPServer. Handles incoming
        requests and renders them through jinja2 if required.
    '''

    CHUNK_SIZE = 1024 * 1024            # read files in 1MB chunks
    MAX_FILE_SIZE = 1024 * 1024 * 50    # upstream will not accept files over 50MB
    server_version = 'SimpleHTTP+GitLayer'
    index = None

    def write_headers(self, response_code=200, mime_type='text/plain'):
        self.send_response(response_code)
        if mime_type.startswith('text/'):
            mime_type += '; charset=utf-8'
        self.send_header('Content-type', mime_type)
        self.end_headers()

    def write(self, s):
        self.wfile.write(force_bytes(s))

    def chunk_read(self, handle):
        '''
            Read a file in defined chunk sizes as a lazy-man's file
            streaming to handle large files.
        '''
        def get_chunk():
            return handle.read(self.CHUNK_SIZE)
        for chunk in iter(get_chunk, ''):
            yield chunk

    def do_GET(self):
        self.index.scan_files()
        actual_path = self.path
        # remove any query strings
        query_pos = actual_path.find('?')
        if query_pos >= 0:
            actual_path = actual_path[:query_pos]
        f = self.index.get_file(actual_path)
        if not f:
            self.do_404()
            return
        # perform sanity checks on all requested files
        if f.size > self.MAX_FILE_SIZE:
            # file is too big to serve on the live platform
            exc_type, exc_value, exc_traceback = sys.exc_info()
            estr = 'File {} is too big! GitLayer will only serve files <= {} bytes'
            e = GitLayerError(estr.format(f.uri, self.MAX_FILE_SIZE))
            e.filename = f.uri
            self.do_500((e, exc_traceback))
            return
        elif (f.do_something) and f.large:
            # file needs some form of rendering but is too large
            exc_type, exc_value, exc_traceback = sys.exc_info()
            estr = 'File {} is too big! GitLayer will only render files <= {} bytes'
            e = GitLayerError(estr.format(f.uri, self.index.MAX_RENDER_SIZE))
            e.filename = f.uri
            self.do_500((e, exc_traceback))
            return
        self.do_200(f)

    def do_200(self, f):
        '''
            Attempt to render the file through filters if it is required, otherwise
            just send the file to the client with the detected MIME type.
        '''
        if f.do_something:
            # file requires some processing
            buffer = render_file(self.index, f, errback=self.do_500)
            if buffer:
                self.write_headers(response_code=200, mime_type=f.mime)
                self.write(buffer)
        elif f.do_concatenation:
            # file is a dynamic concatenation of other files
            previous_file = None
            buffer = ''
            for related_uri in f._related_files:
                related_file = self.index.get_file(related_uri)
                if not related_file:
                    continue
                if not previous_file:
                    previous_file = related_file
                if previous_file.mime != related_file.mime:
                    exc_type, exc_value, exc_traceback = sys.exc_info()
                    e.filename = f.uri
                    self.do_500((e, exc_traceback))
                b = render_file(self.index, related_file, errback=self.do_500)
                if b:
                    buffer += b
            self.write_headers(response_code=200, mime_type=related_file.mime)
            self.write(buffer)
        else:
            # standard file, read it and send it to the client in chunks
            with open(self.index.full_path(f)) as handle:
                self.write_headers(response_code=200, mime_type=f.mime)
                for chunk in self.chunk_read(handle):
                    self.wfile.write(chunk)

    def do_404(self):
        '''
            Attempt to render a custom 404 page if one is present, otherwise fall
            back to a default generic error.
        '''
        custom_404 = self.index.error404
        self.write_headers(response_code=404, mime_type='text/html')
        if custom_404 and not custom_404.large:
            render_file(self.index, custom_404, callback=self.write)
        else:
            self.write('<!doctype html>\n<html>\n<head>\n')
            self.write('<title>404 - Page Not Found</title>\n</head>\n')
            self.write('<body>\n<h1>HTTP 404 - Page Not Found</h1>\n')
            self.write('<p>URI: <strong>{}</strong>'.format(self.path))
            self.write(' does not exist.</p>\n <p>Additionally, a 404 error')
            self.write(' was generated when tryingto handle the 404! Create')
            self.write(' a file called {}'.format(self.index.ERROR_404_FILE))
            self.write(' and make sure it is not too large.</p>\n</body>\n')
            self.write('</html>\n')

    def do_500(self, error):
        '''
            Usually triggered by template errors, display the error with a stack
            trace to the client.
        '''
        e, trace = error
        e.filename = e.filename if e.filename else '&lt;unknown&gt;'
        lineno = getattr(e, 'lineno', '?')
        n = e.__class__.__name__
        log.error('{}:{} - {}: {}'.format(e.filename, lineno, n, e))
        tracest = '<li>{}:{} <h2>{}: {}</h2></li>'.format(e.filename, lineno, n, e)
        tracel = [tracest]
        for call in traceback.extract_tb(trace):
            filename, lineno, method, line = call
            if filename and line:
                filename = filename if filename else '&lt;unknown&gt;'
                thtml = '<li>{}:{}<blockquote>method: {}<br>{}</blockquote></li>'
                tracel.append(thtml.format(filename, lineno, method, line))
        self.write_headers(response_code=500, mime_type='text/html')
        self.write('<!doctype html>\n<html>\n<head>\n')
        self.write('<title>500 - Internal Server Error</title>\n</head>\n')
        self.write('<body>\n<h1>HTTP 500 - Internal Server Error</h1>\n')
        self.write('<pre style="white-space:pre-wrap;word-wrap:break-word;">')
        self.write('\n<p>Your request to: <strong>{}'.format(self.path))
        self.write('</strong> caused an error.')
        self.write(' Backtrace:</p>\n<ul>{}</ul>'.format('\n'.join(tracel)))
        self.write('</pre>\n</body>\n</html>\n')

# eof
