# -*- coding: utf-8 -*-

import os
import sys
import json
from datetime import datetime

from .thirdparty import (dateparser, pytz)
from .errors import GitLayerError

# env detection
####################################################################################

PY3 = int(sys.version[0]) == 3

# helper functions
####################################################################################

class SandboxReader(object):
    '''
        Reads files from disk only from a certian trusted path. Largely redundant
        given the extended existing sandbox, but security is important and this
        has no real negative impact.
    '''

    def __init__(self, root=''):
        self.root = False
        if os.path.isdir(root):
            self.root = root
            if not self.root.endswith(os.sep):
                self.root += os.sep

    def resolve_path(self, path):
        return os.path.abspath(os.path.expandvars(os.path.expanduser(path)))

    def readfile(self, path, encoding='utf-8'):
        '''
            Read the contents of a file and attempt to encode it into unicode,
            otherwise return it as raw bytes.
        '''
        path = self.resolve_path(path)
        if path[:len(self.root)] != self.root:
            raise GitLayerError('file {} outside of root: {}'.format(path, self.root))
        if not os.path.isfile(path):
            raise GitLayerError('file does not exist: {}'.format(path))
        if PY3:
            with open(path, 'r', encoding=encoding) as f:
                return f.read()
        else:
            with open(path, 'r') as f:
                data = f.read()
                try:
                    return data.decode(encoding)
                except:
                    return data

class JsonReader(SandboxReader):
    '''
        Reads a file inside a sandboxed root and then attempts to parse it from
        JSON.
    '''

    MAX_SIZE = 1024*1024    # 1 MB

    def readjson(self, jsonfile):
        if jsonfile.startswith(os.sep):
            jsonfile = jsonfile[1:]
        fullpath = os.path.join(self.root, jsonfile)
        jsonsize = os.path.getsize(fullpath)
        if jsonsize > self.MAX_SIZE:
            raise GitLayerError('JSON file is too big ({} bytes > {} max bytes): {}'
                .format(jsonsize, self.MAX_SIZE, fullpath))
        data = self.readfile(fullpath)
        return json.loads(data)

def bytescale(value, rounding=2, depth=1):
    try:
        value = float(value)
        rounding = int(rounding)
        depth = int(depth)
    except:
        return '0.00'
    rounding = 10 if rounding > 10 else rounding
    rounding = 2 if rounding < 1 else rounding
    depth = 5 if depth > 5 else depth
    depth = 1 if depth < 1 else depth
    for i in range(depth):
        value /= 1024
    fmt = '{{:,.{}f}}'.format(rounding)
    return fmt.format(value)

def file_object_sort(list, key='uri', reverse=False):
    key = str(key)
    reverse = bool(reverse)
    return sorted(list, key=lambda x: getattr(x, key, ''), reverse=reverse)

def soft_parse_date(s):
    if not s:
        return False
    if type(s) == datetime:
        return s
    try:
        return dateparser(s)
    except (TypeError, ValueError, AttributeError) as e:
        return False

def is_string(s):
    return isinstance(s, str) or isinstance(s, unicode)

if PY3:
    string_types = str,
    integer_types = int,
    class_types = type,
    text_type = str
    binary_type = bytes
else:
    string_types = basestring,
    integer_types = (int, long)
    text_type = unicode
    binary_type = str

def force_bytes(s, encoding='utf-8', strings_only=False, errors='strict'):
    if isinstance(s, bytes):
        if encoding == 'utf-8':
            return s
        else:
            return s.decode('utf-8', errors).encode(encoding, errors)
    if not isinstance(s, string_types):
        try:
            if PY3:
                return text_type(s).encode(encoding)
            else:
                return bytes(s)
        except UnicodeEncodeError:
            if isinstance(s, Exception):
                return b' '.join([force_bytes(arg, encoding, strings_only, errors) for arg in s])
            return text_type(s).encode(encoding, errors)
    else:
        return s.encode(encoding, errors)

def ts_to_datetime(ts):
    return pytz.utc.localize(datetime.fromtimestamp(ts))

# eof
