# -*- coding: utf-8 -*-

import os
from shutil import copyfile

from . import log
from .fileindex import GitLayerIndex
from .templating import (jinja_env, render_file)

class BuildSite(object):
    '''
        This class iterates over each file in a site and renders it to the
        output directory.
    '''

    DOT_FILES = ('.hgignore', '.gitignore')
    REPO_DIRS = ('.hg', '.git')

    def __init__(self, site_dir='', output_dir=''):
        self.site_dir = site_dir
        self.output_dir = output_dir
        self.site_root = None
        self.site_dirs = set()
        self.site_files = set()
        self._file_index()
        self._dir_index()

    def _file_index(self):
        for root, dirs, files in os.walk(self.site_dir):
            for repo in self.REPO_DIRS:
                if repo in dirs:
                    if not self.site_root:
                        self.site_root = root
                    dirs.remove(repo)
            for f in files:
                if f in self.DOT_FILES:
                    continue
                self.site_files.add(os.path.join(root, f))

    def _dir_index(self):
        for f in self.site_files:
            if f != self.output_dir:
                self.site_dirs.add(os.path.dirname(f))

    def site_path(self, path):
        return path[:len(self.site_dir)] == self.site_dir

    def output_path(self, path):
        return path[:len(self.output_dir)] == self.output_dir

    def site_to_output_path(self, site_path=''):
        return self.output_dir + site_path[len(self.site_dir):]

    def mkdir(self, path=''):
        if not os.path.isdir(path) and (self.site_path(path) or self.output_path(path)):
            os.makedirs(path)
            return True
        return False

    def is_file(self, path=''):
        # checks that the path is a valid file and that it's not a symlink
        return os.path.isfile(path) and not os.path.islink(path)

    def copy(self, from_path='', to_path=''):
        if self.site_path(from_path) and self.output_path(to_path) and self.is_file(from_path) and not self.is_file(to_path):
            copyfile(from_path, to_path)
            return True
        return False

    def build(self):
        # create any required target directories
        for d in self.site_dirs:
            self.mkdir(self.site_to_output_path(d))
        # build the index of the working directory
        index = GitLayerIndex(self.site_dir)
        # set up jinja2
        env = jinja_env(self.site_dir)
        # iterate over the index
        for f in index.all_files_and_errors:
            # file source and destination
            site_path = index.full_path(f)
            if site_path:
                output_path = self.site_to_output_path(site_path)
            if f.do_concatenation:
                # concat two files of the same MIME together
                concat_files = '[' + ','.join(f._related_files) + ']'
                previous_file = None
                render_error = False
                for related_uri in f._related_files:
                    related_file = index.get_file(related_uri)
                    if not related_file:
                        continue
                    if not previous_file:
                        previous_file = related_file
                    if previous_file.mime != related_file.mime:
                        estr = 'can only concat files of the same mime! {} ({}) != {} ({})'
                        log.error(estr.format(related_file.uri, related_file.mime,
                            previous_file.uri, previous_file.mime))
                        render_error = True
                    fh = open(output_path, 'a+')
                    render_file(index, related_file, callback=fh.write,
                        errback=log.error)
                if not render_error:
                    log.info('concat {}{} -> {}'.format(self.site_dir,
                        concat_files, output_path))
            elif f.do_something:
                # requires some form of postprocess rendering
                fh = open(output_path, 'w+')
                render_file(index, f, callback=fh.write, errback=log.error)
                log.info('render {} -> {} [{}]'.format(site_path, output_path,
                    f.do_something))
            else:
                # normal file, straight copy it
                copyfile(site_path, output_path)
                log.info('copied {} -> {}'.format(site_path, output_path))
        if log.all_errors:
            log.critical('There were errors building your site! These were:')
            for e in log.all_errors:
                log.critical(e)

# eof
