# -*- coding: utf-8 -*-

import os
import traceback
import unicodedata
import string
from copy import copy
from datetime import datetime
from fnmatch import fnmatch
from collections import OrderedDict

from . import log
from .errors import GitLayerError
from .fileobj import (GitLayerFile, GitLayerBlocks)
from .utils import (SandboxReader, file_object_sort, is_string, force_bytes,
    soft_parse_date)
from .locales import localeman
from .thirdparty import pytz
from .templating import (jinja_env, blank_context, MarkdownRenderer)

class GitLayerIndex(object):
    '''
        Populates, stores and indexes the GitLayerFile instances.
    '''

    HTML_EXTENSION = 'html'
    MINIFY_FILE_PART = 'min'
    SCSS_FILE_PART = 'scss'
    JINJA_RENDER_MIME = 'text/html'
    SCSS_RENDER_MIME = 'text/css'
    MARKDOWN_RENDER_MIME = 'text/x-markdown'
    MAX_RENDER_SIZE = 1024 * 100    # don't attempting to render html files >100KB
    IGNORE_DIRS = ('.hg', '.git')
    IGNORE_FILES = ('gitlayer.py', '.hgignore', '.gitignore')
    INDEX_FILES = ('/index.html', '/index.htm')
    ERROR_401_FILE = '/401error.html'
    ERROR_404_FILE = '/404error.html'
    SITEMAP_FILE = '/sitemap.xml'
    DEFAULT_MIME = 'text/plain'     # this is only for the development server
    MIME_TYPES = (
        ('application/atom+xml', ('atom',)),
        ('application/java-archive', ('jar',)),
        ('application/java-vm', ('class',)),
        ('application/javascript', ('js',)),
        ('application/json', ('json',)),
        ('application/octet-stream', ('bin',)),
        ('application/ogg', ('ogx',)),
        ('application/pdf', ('pdf',)),
        ('application/pgp-encrypted', ('pgp',)),
        ('application/pgp-keys', ('key',)),
        ('application/pgp-signature', ('sig',)),
        ('application/xhtml+xml', ('xhtml',)),
        ('application/xml', ('xml', 'xsl', 'xsd')),
        ('application/zip', ('zip',)),
        ('application/vnd.mozilla.xul+xml', ('xul',)),
        ('application/vnd.ms-excel', ('xls', 'xlb', 'xlt')),
        ('application/vnd.ms-fontobject', ('eot',)),
        ('application/x-7z-compressed', ('7z',)),
        ('application/x-bittorrent', ('torrent',)),
        ('application/x-cab', ('cab',)),
        ('application/x-font-woff', ('woff',)),
        ('application/x-gtar-compressed', ('tgz', 'taz')),
        ('application/x-latex', ('latex',)),
        ('application/x-python-code', ('pyc', 'pyo')),
        ('application/x-rss+xml', ('rss',)),
        ('application/x-ruby', ('rb',)),
        ('application/x-shockwave-flash', ('swf', 'swfl',)),
        ('application/x-silverlight', ('scr',)),
        ('application/x-sql', ('sql',)),
        ('application/x-tar', ('tar',)),
        ('application/x-tcl', ('tcl',)),
        ('application/x-x509-ca-cert', ('crt',)),
        ('application/x-xpinstall', ('xpi',)),
        ('audio/basic', ('au', 'snd')),
        ('audio/flac', ('flac',)),
        ('audio/midi', ('mid', 'midi', 'kar',)),
        ('audio/mpeg', ('mpga', 'mpega', 'mp2', 'mp3', 'm4a')),
        ('audio/mpegurl', ('m3u',)),
        ('audio/ogg', ('oga', 'ogg', 'spx')),
        ('audio/x-mpegurl', ('m3u',)),
        ('audio/x-ms-wma', ('wma',)),
        ('audio/x-wav', ('wav',)),
        ('image/gif', ('gif',)),
        ('image/jpeg', ('jpeg', 'jpg', 'jpe')),
        ('image/png', ('png',)),
        ('image/svg+xml', ('svg', 'svgz')),
        ('image/tiff', ('tiff', 'tif')),
        ('image/vnd.microsoft.icon', ('ico',)),
        ('image/vnd.wap.wbmp', ('wbmp',)),
        ('image/x-ms-bmp', ('bmp',)),
        ('message/rfc822', ('eml',)),
        ('text/css', ('css',)),
        ('text/csv', ('csv',)),
        ('text/html', ('html', 'htm', 'shtml')),
        ('text/plain', ('asc', 'txt', 'text', 'pot', 'brf', 'srt')),
        ('text/x-c++src', ('c++', 'cpp', 'cxx', 'cc')),
        ('text/x-chdr', ('h',)),
        ('text/x-csrc', ('c',)),
        ('text/x-dsrc', ('d',)),
        ('text/x-diff', ('diff', 'patch')),
        ('text/x-haskell', ('hs',)),
        ('text/x-java', ('java',)),
        ('text/x-markdown', ('md', 'markdown')),
        ('text/x-pascal', ('p', 'pas')),
        ('text/x-perl', ('pl', 'pm')),
        ('text/x-python', ('py',)),
        ('text/x-scala', ('scala',)),
        ('text/x-sh', ('sh',)),
        ('text/x-tcl', ('tcl', 'tk')),
        ('text/x-tex', ('tex', 'ltx', 'sty', 'cls')),
        ('video/3gpp', ('3gp',)),
        ('video/mpeg', ('mpeg', 'mpg', 'mpe')),
        ('video/mp4', ('mp4',)),
        ('video/quicktime', ('qt', 'mov')),
        ('video/ogg', ('ogv',)),
        ('video/webm', ('webm',)),
        ('video/x-flv', ('flv',)),
        ('video/x-ms-asf', ('asf', 'asx')),
        ('video/x-ms-wm', ('wm',)),
        ('video/x-ms-wmv', ('wmv',)),
        ('video/x-msvideo', ('avi',)),
        ('video/x-sgi-movie', ('movie',)),
        ('video/x-matroska', ('mpv', 'mkv')),
    )
    MIME_REMAPS = {
        'text/x-markdown': 'text/html',
    }

    def __init__(self, path=''):
        if not os.path.isdir(path):
            raise GitLayerError('path is not a directory: {}'.format(path))
        self.path = os.path.abspath(os.path.normpath(path))
        self.readfile = SandboxReader(self.path).readfile
        self._files = {}            # dict of GitLayerFile instanced indexed by URI
        self._modtime_cache = {}    # dict of os.stat.mtime() indexed by full path
        self.initial_scan = True
        self.scan_files()
        self.initial_scan = False

    def _get_mime_type(self, ext):
        '''
            Returns the MIME type for a file extension or the default
            MIME type if it doesn't match.
        '''
        ext = str(ext)
        for mime, exts in self.MIME_TYPES:
            if ext in exts:
                return mime
        return self.DEFAULT_MIME

    def scan_files(self):
        '''
            os.walk()'s over the web root and checks each files mtime to see if it
            needs reloading. Files are lazily reloaded when their mtime changes
            (which implies the file has been modified).
        '''
        files_updated = False
        for root, dirs, files in os.walk(self.path):
            for ignore_dir in self.IGNORE_DIRS:
                if ignore_dir in dirs:
                    dirs.remove(ignore_dir)
            for ignore_file in self.IGNORE_FILES:
                if ignore_file in files:
                    files.remove(ignore_file)
            for file_name in files:
                full_path = os.path.join(root, file_name)
                modtime = os.path.getmtime(full_path)
                if modtime > self._modtime_cache.get(full_path, 0):
                    files_updated = True
                    self._scan_file(modtime, root, file_name, full_path)
        if files_updated:
            for uri,f in self._files.iteritems():
                if f.parent_uri:
                    log.info('detected parent: {} for file: {}'.format(f.parent_uri, f.uri))
                    f.parent = self.get_file(f.parent_uri)
            self.generate_sitemap()

    def _scan_file(self, modtime, root, file_name, full_path):
        uri = full_path[len(self.path):]
        uri = '/'.join(uri.split(os.sep))
        if not self.initial_scan:
            log.info('mtime changed, updating: {}'.format(uri))
        self._modtime_cache[full_path] = modtime
        self._files[uri] = self._store_file(full_path, root, file_name, modtime, uri)
        if self._files[uri].is_base and not self.initial_scan:
            self._handle_template_file(self._files[uri])
            # no further processing as a full rescap is being performed
            return
        elif self._files[uri].is_remap:
            self._handle_remap_file(self._files[uri])
        if uri in self.INDEX_FILES:
            # store the same file again under the root uri
            self._files['/'] = copy(self._files[uri])
            self._files['/'].is_alias = True

    def _handle_template_file(self, f):
        log.info('changed file is a base template, reloading cache')
        self.initial_scan = True
        # clear modtimes for all files that are not this file
        self._modtime_cache = {}
        self._modtime_cache[self.full_path(f)] = f._modtime
        self._files = {}
        self._files[f.uri] = f
        # force a rescan of all files
        self.scan_files()
        self.initial_scan = False

    def _handle_remap_file(self, f):
        # file has been remapped, change its URI
        remapped_uri = f.remapped_uri
        if remapped_uri not in self._files or f.original_uri == f.uri:
            # only change the URI if another file hasn't used it
            log.info('remapped {} -> {}'.format(f.uri, remapped_uri))
            self._files[remapped_uri] = copy(self._files[f.uri])
            self._files[remapped_uri].uri = remapped_uri
            del self._files[f.uri]
        else:
            log.error('failed to remap {} -> {}, uri already in use'
                .format(f.uri, remapped_uri))

    def _store_file(self, full_path, root, file_name, modtime, uri):
        '''
            Create and store a GitLayerFile instance.
        '''
        name_parts = file_name.split('.')
        file_name_parts = name_parts[:-1]
        name = '.'.join(file_name_parts)
        ext = name_parts[-1].lower()
        do_minify = False
        do_scss = False
        for p in file_name_parts:
            if p == self.MINIFY_FILE_PART:
                do_minify = True
            elif p == self.SCSS_FILE_PART:
                do_scss = True
        f = GitLayerFile()
        f.uri = uri
        f.original_uri = uri
        f.file = os.path.basename(full_path)
        f.dir = os.path.dirname(uri)
        f.name = name
        f.nameparts = f.name.split('-')
        f.ext = ext
        f.size = os.path.getsize(full_path)
        f.large = f.size > self.MAX_RENDER_SIZE
        f.mime = self._get_mime_type(ext)
        f.do_jinja = f.mime == self.JINJA_RENDER_MIME
        f.do_scss = do_scss
        f.do_minify = do_minify
        f.do_markdown = f.mime == self.MARKDOWN_RENDER_MIME
        f.is_error401 = uri == self.ERROR_401_FILE
        f.is_error404 = uri == self.ERROR_404_FILE
        f.is_base = name.startswith('_')
        f._modtime = modtime
        f.is_html = f.do_jinja and not f.is_error401 and not f.is_error404 \
            and not f.is_base
        if (f.do_markdown or f.do_jinja) and f.large:
            # file is too large to bother rendering
            estr = 'error! not pre-rendering file as it is too large: {}'
            log.error(estr.format(f.uri))
            return
        markdown_and_jinja = False
        if f.do_markdown:
            # pre-render with the markdown renderer
            self.initial_markdown_render(f, full_path)
        if f.do_jinja:
            # pre-render the template to extract sharable data
            self.initial_jinja_render(f, full_path)
        # remap mime if required
        f.mime = self.MIME_REMAPS.get(f.mime, f.mime)
        return f

    def _concat_file(self, uri, files):
        '''
            Create a special dynamic concatenated file object.
        '''
        if not uri or uri in self._files:
            return
        ext = uri.split('.')[-1]
        mime = self._get_mime_type(ext)
        for f in files:
            f_ext = f.split('.')[-1]
            f_mime = self._get_mime_type(f_ext)
            if mime != f_mime:
                estr = 'error concatenating files! differing MIME types: {}'
                log.error(estr.format(', '.join(files)))
                return
        self._files[uri] = GitLayerFile()
        self._files[uri].uri = uri
        self._files[uri].ext = ext
        self._files[uri].do_concatenation = True
        self._files[uri]._related_files = files
        log.info('concatenated: {} -> {}'.format(', '.join(files), uri))

    def initial_jinja_render(self, f, full_path):
        '''
            Perform an initial double rendering of a file with no context to extract
            any present metadata, then a rendering with no neighbour context. Write
            rendering errors to stdout but continue running.
        '''
        # create the jinja2 environment
        env = jinja_env(self.path)
        template_data = f.content if f.dynamic else self.readfile(full_path)
        # first render, extract metadata
        try:
            metadata_template = env.from_string(template_data)
            metadata_template.render(**blank_context(f))
            f._groups = f._groups | env.template_groups
            f._tags = f._tags | env.template_tags
            if env.template_timezone:
                f.timezone = env.template_timezone
            f.date = f.localise_date(env.template_date)
            if f.date:
                try:
                    f.utcdate = f.date.astimezone(pytz.utc)
                except ValueError:
                    f.utcdate = f.date.replace(tzinfo=pytz.utc)
            if env.template_locale:
                f.locale = env.template_locale
                f.locale_changed = True
            if env.template_uri:
                f.remapped_uri = env.template_uri
                f.is_remap = True
            if env.template_parenturi:
                f.parent_uri = env.template_parenturi
            for concat_uri, concat_files in env.template_concat.items():
                if concat_uri not in self._files:
                    self._concat_file(concat_uri, concat_files)
        except Exception as e:
            trace = traceback.format_exc()
            e.filename = f.uri
            estr = 'error extracting metadata! detected error in template: {}'
            log.error(estr.format(f.uri))
            lineno = getattr(e, 'lineno', '?')
            log.error('{} in: {} on line: {}'.format(e, f.uri, lineno))
            log.error(trace)
            return False
        # second render, block extraction
        bc = blank_context(f)
        data_template = env.from_string(template_data)
        if f.locale_changed:
            localeman.switch(f.locale)
        for block_n, block_g in data_template.blocks.items():
            try:
                for b in block_g(data_template.new_context(vars=bc)):
                    setattr(f.blocks, block_n, getattr(f.blocks, block_n, '') + b)
            except Exception as e:
                estr = 'not rendering block {}:{} - requires parent context'
                log.info(estr.format(f.uri, block_n))
                setattr(f.blocks, block_n, getattr(f.blocks, block_n, '') + '')
        localeman.restore()
        return True

    def initial_markdown_render(self, f, full_path):
        '''
            Perform an initial rendering of the markdown file. If the file has a
            yaml header parse it and then create a Jinja template from it to be
            re-rendered in Jinja. This allows for template extensions from pure
            markdown files.
        '''
        template_data = self.readfile(full_path)
        md = MarkdownRenderer(template_data, silent=True)
        header, body = md.header, md.body
        template = u''
        contentblock = 'content'
        template_uri = None
        template_title = None
        template_date = None
        if type(header) == dict and header:
            # markdown file has a yaml header with valid data, create a bare
            # template around it and set jinja to re-pre-render it
            for k,v in header.items():
                k = k.lower().strip()
                b = k.split('.')
                if k == 'extends':
                    template += u'{{% extends \'{}\' %}}\n'.format(v)
                elif k == 'title':
                    template_title = v
                    template += u'{{% block title %}}\n{}\n{{% endblock %}}\n'.format(v)
                elif k == 'uri':
                    template_uri = v
                    template += u'{{% uri \'{}\' %}}\n'.format(v)
                elif k == 'date':
                    template_date = v
                    template += u'{{% date \'{}\' %}}\n'.format(v.isoformat())
                elif k == 'locale':
                    template += u'{{% locale \'{}\' %}}\n'.format(v)
                elif k == 'timezone':
                    template += u'{{% timezone \'{}\' %}}\n'.format(v)
                elif k == 'parent':
                    template += u'{{% parent \'{}\' %}}\n'.format(v)
                elif k == 'tags':
                    template += u'{{% tags %}}{}{{% endtags %}}\n'.format(v)
                elif k == 'groups':
                    template += u'{{% groups %}}{}{{% endgroups %}}\n'.format(v)
                elif k == 'contentblock':
                    contentblock = str(v)
                elif len(b) > 1 and b[0] == 'block':
                    template += u'{{ % block {} %}}\n{}\n{{% endblock %}}\n'.format(b[1], v)
        template += u'{{% block {} %}}\n{}\n{{% endblock %}}\n'.format(contentblock, body)
        if not template_uri and template_title:
            template_uri = template_date.strftime('%Y-%m-%d_') if template_date else u''
            for c in unicodedata.normalize('NFD', force_bytes(template_title).decode('utf-8')):
                if c in string.lowercase:
                    template_uri += c
                elif c == ' ':
                    template_uri += ' '
            template_uri = template_uri.strip()
            template_uri = template_uri.replace(' ', '-')
            template += u'{{% uri \'/{}.html\' %}}\n'.format(template_uri)
        f.content = template
        f.do_jinja = True
        f.do_markdown = False
        return True

    def generate_sitemap(self):
        existing_sitemap = self._files.get(self.SITEMAP_FILE)
        if existing_sitemap and not existing_sitemap.dynamic:
            return
        sitemap = GitLayerFile()
        sitemap.file = os.path.basename(self.SITEMAP_FILE)
        sitemap.date = pytz.utc.localize(datetime.utcnow())
        filename, ext = sitemap.file.split('.')
        sitemap.dir = os.path.dirname(self.SITEMAP_FILE)
        sitemap.uri = self.SITEMAP_FILE
        sitemap.name = filename
        sitemap.nameparts = filename.split('-')
        sitemap.ext = ext
        sitemap.size = 0
        sitemap.mime = self._get_mime_type(ext)
        content = ['<?xml version="1.0" encoding="UTF-8"?>']
        content.append('<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">')
        for page in self.all_pages:
            pagedate = self.file_utc_modtime(page)
            datestr = pagedate.strftime('%Y-%m-%dT%H:%M:%S%z')
            datestr = datestr[22:] + ':' + datestr[:22]
            content.append('\t<sitemap>')
            content.append('\t\t<loc>{}</loc>'.format(page.uri))
            content.append('\t\t<lastmod>{}</lastmod>'.format(datestr))
            content.append('\t</sitemap>')
        content.append('</sitemapindex>')
        sitemap.content = '\r\n'.join(content)
        self._files[self.SITEMAP_FILE] = sitemap

    def get_file(self, request, default=False):
        '''
            Get a GitLayerFile instance by URI (or return a default)
        '''
        return self._files.get(request, default)

    def full_path(self, f=None):
        '''
            Returns the full local path to a file if the URI exists.
        '''
        uri = f.original_uri if f.is_alias else f.uri
        uri = uri[1:] if uri.startswith('/') else uri
        return os.path.join(self.path, uri)

    def file_utc_modtime(self, f):
        if f.date:
            if not f.date.tzinfo:
                d = f.date.replace(tzinfo=pytz.utc)
            else:
                d = f.date.astimezone(pytz.utc)
        else:
            if f.uri in self._modtime_cache:
                d = ts_to_datetime(self._modtime_cache[f.uri])
            else:
                d = pytz.utc.localize(datetime.utcnow())
        return d

    def files_datesort(self, files=[], reverse=False):
        '''
            Sort a list of files by date.
        '''
        return sorted(files, key=lambda f: f.date, reverse=reverse)

    def files_urisort(self, files=[], reverse=False):
        '''
            Sort a list of files by URI.
        '''
        return sorted(files, key=lambda f: f.uri, reverse=reverse)

    @property
    def error404(self):
        '''
            Return a GitLayerFile instance if one exists with the error404
            default URI, otherwise False.
        '''
        return self.get_file(self.ERROR_404_FILE)

    @property
    def error401(self):
        '''
            Return a GitLayerFile instance if one exists with the error404
            default URI, otherwise False.
        '''
        return self.get_file(self.ERROR_401_FILE)

    @property
    def all_groups(self):
        '''
            Returns all groups from every file as a sorted list.
        '''
        groups = set()
        for path, f in self._files.items():
            groups = groups | f._groups
        return sorted(list(groups))

    @property
    def all_tags(self):
        '''
            Returns all tags from every file as a sorted list.
        '''
        tags = set()
        for path, f in self._files.items():
            tags = tags | f._tags
        return sorted(list(tags))

    @property
    def all_files_and_errors(self):
        '''
            Returns all the files in an unsorted list by their URI including error
            pages.
        '''
        return list([f for f in self._files.values() if not f.is_alias])

    @property
    def all_files(self):
        '''
            Returns all files as a list sorted by their URI excudling aliases and
            the 404 page.
        '''
        return self.files_urisort([f for f in self._files.values() if not f.is_alias
            and not f.is_error404 and not f.is_error401])

    @property
    def all_pages(self):
        '''
            Returns all HTML files as a list sorted by their URI.
        '''
        return self.files_urisort([f for f in self._files.values() if f.is_html and
            not f.is_alias])

    @property
    def all_posts(self):
        '''
            Returns all posts (pages that have a date) sorted by their date.
        '''
        return self.files_datesort([f for f in self._files.values() if f.is_html
            and f.date])

    def file_match_by_groups(self, groups=[]):
        '''
            Returns a list of GitLayerFile instances sorted by URI
            that match the provided set of groups.
        '''
        groups = [groups] if is_string(groups) else groups
        groups = set(groups)
        matches = set()
        for f in self._files.values():
            if f._groups & groups and f.is_html and not f.is_alias:
                matches.add(f)
        return self.files_urisort(list(matches))

    def file_match_by_tags(self, tags=[]):
        '''
            Returns a list of GitLayerFile instances sorted by URI
            that match the provided set of tags.
        '''
        tags = [tags] if is_string(tags) else tags
        tags = set(tags)
        matches = set()
        for f in self._files.values():
            if f._tags & tags and f.is_html and not f.is_alias:
                matches.add(f)
        return self.files_urisort(list(matches))

    def file_match_by_uri(self, uri):
        '''
            Returns a list of GitLayerFile instances sorted by URI
            that match the provided URI prefix.
        '''
        uri = str(uri)
        matches = set()
        for f in self._files.values():
            if not f.is_alias and fnmatch(f.uri, uri):
                matches.add(f)
        return self.files_urisort(list(matches))

    def posts_into_datetree(self, posts=[], reverse=True):
        '''
            Returns a tree of all posts in reverse date order in the
            following OrderedDict tree format:
            OrderedDict{
                year: {
                    month: {
                        day: {
                            [list of posts]
                        }
                    }
                }
            }
        '''
        posts = posts if posts else self.all_posts
        posts = reversed(posts) if reverse else posts
        tree = OrderedDict()
        for post in posts:
            if post.date:
                y, m, d = post.date.year, post.date.month, post.date.day
                dy = datetime(year=y, month=1, day=1)
                dm = datetime(year=y, month=m, day=1)
                dd = datetime(year=y, month=m, day=d)
                tree.setdefault(dy, OrderedDict()) \
                    .setdefault(dm, OrderedDict()) \
                    .setdefault(dd, []).append(post)
        return tree

    def posts_by_daterange(self, posts=[], date_from=None, date_to=None, days=0):
        '''
            Returns any posts that fall between the two supplied dates.
        '''
        posts = posts if posts else reversed(self.all_posts)
        date_from = soft_parse_date(date_from)
        date_to = soft_parse_date(date_to)
        try:
            days = int(days)
        except (TypeError, ValueError):
            days = 0
        if not date_from and not date_to and days == 0:
            return posts
        if date_to and days > 0:
            return posts
        if date_from and not date_from.tzinfo:
            date_from = date_from.replace(tzinfo=pytz.utc)
        date_to = datetime.utcnow() - timedelta(days=days) if days > 0 else date_to
        if not date_to.tzinfo:
            date_to = date_to.replace(tzinfo=pytz.utc)
        if date_from and date_to:
            date_to, date_from = min(date_from, date_to), max(date_from, date_to)
            return [f for f in posts if f.utcdate <= date_from
                and f.utcdate >= date_to]
        else:
            return [f for f in posts if f.utcdate >= date_to]

    def get_neighbour_post(self, post, groups=[], tags=[], direction=1):
        groups = [groups] if is_string(groups) else groups
        groups = set(groups)
        tags = [tags] if is_string(tags) else tags
        tags = set(tags)
        matches = []
        for p in self.all_posts:
            pgroups = set(p.groups)
            ptags = set(p.tags)
            if not groups and not tags:
                matches.append(p)
            elif groups and tags and pgroups & groups and ptags & tags:
                matches.append(p)
            elif groups and pgroups & groups:
                matches.append(p)
            elif tags and ptags & tags:
                matches.append(p)
        try:
            current = matches.index(post)
        except ValueError:
            return None
        nextpos = current + direction
        if nextpos < 0 or nextpos >= len(matches):
            return None
        try:
            return matches[current + direction]
        except IndexError:
            return None

    def next_post(self, post, groups=[], tags=[]):
        return self.get_neighbour_post(post, groups, tags, direction=1)

    def prev_post(self, post, groups=[], tags=[]):
        return self.get_neighbour_post(post, groups, tags, direction=-1)

# eof
