# -*- coding: utf-8 -*-

import os
import json
import unittest
import tempfile
from datetime import datetime

from gitlayer import (utils, errors)

class TestUtils(unittest.TestCase):

    def test_sandbox_reader(self):
        d = os.urandom(32)
        with tempfile.NamedTemporaryFile() as f:
            f.write(d)
            f.flush()
            dname = os.path.dirname(f.name)
            sbr = utils.SandboxReader(dname)
            self.assertEqual(sbr.root, dname + os.sep)
            with self.assertRaises(errors.GitLayerError):
                sbr.readfile('/gitlayer/' + d.encode('hex'))
            with self.assertRaises(errors.GitLayerError):
                sbr.readfile(dname)
            self.assertEqual(sbr.readfile(f.name), d)
        if os.path.exists(f.name):
            os.path.unlink(f.name)

    def test_json_reader(self):
        d = {'git':'layer'}
        with tempfile.NamedTemporaryFile() as f:
            f.write(json.dumps(d))
            f.flush()
            dname = os.path.dirname(f.name)
            fname = os.path.basename(f.name)
            jsr = utils.JsonReader(dname)
            self.assertEqual(jsr.readjson(fname), d)
        if os.path.exists(f.name):
            os.path.unlink(f.name)

    def test_bytescale(self):
        self.assertEqual(utils.bytescale(1024, rounding=3, depth=1), '1.000')
        self.assertEqual(utils.bytescale(1048576, rounding=2, depth=2), '1.00')
        self.assertEqual(utils.bytescale(1073741824, rounding=1, depth=3), '1.0')

    def test_file_object_sort(self):
        class f(object):
            pass
        f1, f2, f3, f4, f5 = f(), f(), f(), f(), f()
        f1.p, f2.p, f3.p, f4.p, f5.p = 'x', 'u', 's', 'q', 'a'
        f = [f1, f2, f3, f4, f5]
        s1 = [x.p for x in utils.file_object_sort(f, key='p')]
        self.assertEqual(s1, ['a', 'q', 's', 'u', 'x'])
        s2 = [x.p for x in utils.file_object_sort(f, key='p', reverse=True)]
        self.assertEqual(s2, ['x', 'u', 's', 'q', 'a'])

    def test_soft_parse_date(self):
        d1 = utils.soft_parse_date('1970-01-01')
        self.assertEqual(d1.year, 1970)
        self.assertEqual(d1.month, 1)
        self.assertEqual(d1.day, 1)
        d2 = utils.soft_parse_date('Thursday 1st Jan 1970')
        self.assertEqual(d2.year, 1970)
        self.assertEqual(d2.month, 1)
        self.assertEqual(d2.day, 1)
        d3 = utils.soft_parse_date('1970-01-01 00:00')
        self.assertEqual(d3.year, 1970)
        self.assertEqual(d3.month, 1)
        self.assertEqual(d3.day, 1)

    def test_is_string(self):
        self.assertTrue(utils.is_string(''))
        self.assertTrue(utils.is_string(u''))
        self.assertFalse(utils.is_string(0))

    def test_force_bytes(self):
        self.assertEqual(utils.force_bytes('\xe2\x98\x83'.decode('utf-8')), '\xe2\x98\x83')
        self.assertEqual(utils.force_bytes('a'), 'a')
        self.assertEqual(utils.force_bytes(u'a'), 'a')

    def test_ts_to_datetime(self):
        d = utils.ts_to_datetime(0)
        self.assertEqual(d.year, 1970)
        self.assertEqual(d.month, 1)
        self.assertEqual(d.day, 1)

# eof