# -*- coding: utf-8 -*-

import os
import shutil
import tempfile
import unittest

from gitlayer import build

class TestBuild(unittest.TestCase):

    def setUp(self):
        self.data = os.urandom(32)
        self.fakesite = tempfile.mkdtemp()
        self.builddir = tempfile.mkdtemp()
        os.mkdir(os.path.join(self.fakesite, 'b'))
        os.mkdir(os.path.join(self.fakesite, 'c'))
        os.mkdir(os.path.join(self.fakesite, 'd'))
        self.fakefiles = []
        f = os.path.join(self.fakesite, 'a.html')
        open(f, 'w').write('{% groups %}a, b{% endgroups %}{% tags %}c, d{% endtags %}{% date "1970-01-01" %}{{ this.uri }}')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'b.html')
        open(f, 'w').write('{% date "1970-01-02" %}')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'c.html')
        open(f, 'w').write('{% concat "/a.html", "/p.html" %}')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, '401error.html')
        open(f, 'w').write('<p>error 401</p>')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, '404error.html')
        open(f, 'w').write('<p>error 404</p>')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, '_t.html')
        open(f, 'w').write('<p>template</p>')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'b/b.scss')
        open(f, 'w').write('.a { .b { .c { git: layer; } } }')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'c/c.md')
        open(f, 'w').write('# gitlayer')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'd/d.jpg')
        open(f, 'w').write(self.data)
        self.fakefiles.append(f)
        self.sitemapfile = '/sitemap.xml'
        self.concatfile = '/20b147f91fd109b1d473.html'
        self.defaultstate = [self.sitemapfile, self.concatfile]
        self.builder = build.BuildSite(site_dir=self.fakesite, output_dir=self.builddir)

    def test_build_dircheck(self):
        self.assertEqual(self.builder.site_dir, self.fakesite)
        self.assertEqual(self.builder.output_dir, self.builddir)
        d = sorted([os.path.join(self.fakesite, x) for x in ['b','c','d']])
        d.insert(0, self.fakesite)
        self.assertEqual(sorted(self.builder.site_dirs), d)
        self.assertEqual(sorted(self.builder.site_files), sorted(self.fakefiles))

    def test_site_path(self):
        self.assertTrue(self.builder.site_path(self.fakesite))

    def test_output_path(self):
        self.assertTrue(self.builder.output_path(self.builddir))

    def test_site_to_output_path(self):
        i = os.path.join(self.fakesite, 'gitlayer')
        o = os.path.join(self.builddir, 'gitlayer')
        self.assertEqual(self.builder.site_to_output_path(i), o)

    def test_safe_mkdir(self):
        inside_site = os.path.join(self.builddir, self.data.encode('hex'))
        outside_site = os.path.join(tempfile.gettempdir(), self.data.encode('hex'))
        self.assertTrue(self.builder.mkdir(inside_site))
        self.assertFalse(self.builder.mkdir(outside_site))

    def test_safe_isfile(self):
        a_file = os.path.join(self.fakesite, 'a.html')
        not_a_file = os.path.join(self.fakesite, 'z.html')
        self.assertTrue(self.builder.is_file(a_file))
        self.assertFalse(self.builder.is_file(not_a_file))

    def test_safe_copy(self):
        s = os.path.join(self.fakesite, 'a.html')
        d = os.path.join(self.builddir, 'a.html')
        f = os.path.join(tempfile.gettempdir(), self.data.encode('hex'))
        self.assertTrue(self.builder.copy(s, d))
        self.assertFalse(self.builder.copy(s, f))

    def tearDown(self):
        shutil.rmtree(self.fakesite)

# eof