# -*- coding: utf-8 -*-

import unittest
import tempfile
import locale
from hashlib import sha512
from datetime import datetime

from gitlayer import (templating, thirdparty, fileobj)

class TestTemplating(unittest.TestCase):

    def test_datetimeformat_filter(self):
        d = datetime.utcfromtimestamp(0)
        self.assertEqual(templating.datetimeformat_filter(d), '1970-01-01 00:00:00')

    def test_bytestokilo_filter(self):
        self.assertEqual(templating.bytestokilo_filter(1024), '1.00')
        self.assertEqual(templating.bytestokilo_filter(1048576), '1,024.00')
        self.assertEqual(templating.bytestokilo_filter(1073741824), '1,048,576.00')

    def test_bytestomega_filter(self):
        self.assertEqual(templating.bytestomega_filter(1048576), '1.00')
        self.assertEqual(templating.bytestomega_filter(1073741824), '1,024.00')
        self.assertEqual(templating.bytestomega_filter(1099511627776), '1,048,576.00')

    def test_jinja_render(self):
        t = '{{ git }}'
        c = {'git':'layer'}
        e = templating.jinja_env()
        self.assertEqual(templating.jinja_render(t, context_dict=c, env=e), c.get('git'))

    def test_jinja_groups_extension(self):
        d = '{% groups %}git, layer{% endgroups %}'
        e = thirdparty.SandboxedEnvironment(
            extensions=[templating.JinjaGroupsExtension],
            loader=thirdparty.FileSystemLoader(tempfile.gettempdir()))
        t = e.from_string(d)
        self.assertEqual(t.render(), '')
        self.assertEqual(sorted(list(e.template_groups)), ['git', 'layer'])

    def test_jinja_tags_extension(self):
        d = '{% tags %}git, layer{% endtags %}'
        e = thirdparty.SandboxedEnvironment(
            extensions=[templating.JinjaTagsExtension],
            loader=thirdparty.FileSystemLoader(tempfile.gettempdir()))
        t = e.from_string(d)
        self.assertEqual(t.render(), '')
        self.assertEqual(sorted(list(e.template_tags)), ['git', 'layer'])

    def test_jinja_date_extension(self):
        d = '{% date "1970-01-01 00:00:00" %}'
        e = thirdparty.SandboxedEnvironment(
            extensions=[templating.JinjaDateExtension],
            loader=thirdparty.FileSystemLoader(tempfile.gettempdir()))
        t = e.from_string(d)
        self.assertEqual(t.render(), '')
        self.assertEqual(e.template_date.year, 1970)
        self.assertEqual(e.template_date.month, 1)
        self.assertEqual(e.template_date.day, 1)

    def test_jinja_locale_extension(self):
        l = '.'.join([str(l) for l in locale.getdefaultlocale() if l])
        d = '{{% locale "{}" %}}'.format(l)
        e = thirdparty.SandboxedEnvironment(
            extensions=[templating.JinjaLocaleExtension],
            loader=thirdparty.FileSystemLoader(tempfile.gettempdir()))
        t = e.from_string(d)
        self.assertEqual(t.render(), '')
        self.assertEqual(e.template_locale, l)

    def test_jinja_timezone_extension(self):
        d = '{% timezone "utc" %}'
        e = thirdparty.SandboxedEnvironment(
            extensions=[templating.JinjaTimezoneExtension],
            loader=thirdparty.FileSystemLoader(tempfile.gettempdir()))
        t = e.from_string(d)
        self.assertEqual(t.render(), '')
        self.assertEqual(e.template_timezone, thirdparty.pytz.UTC)

    def test_jinja_uri_extension(self):
        d = '{% uri "/git layer.html" %}'
        e = thirdparty.SandboxedEnvironment(
            extensions=[templating.JinjaUriExtension],
            loader=thirdparty.FileSystemLoader(tempfile.gettempdir()))
        t = e.from_string(d)
        self.assertEqual(t.render(), '')
        self.assertEqual(e.template_uri, '/git%20layer.html')

    def test_jinja_parent_extension(self):
        d = '{% parent "/git layer.html" %}'
        e = thirdparty.SandboxedEnvironment(
            extensions=[templating.JinjaParentExtension],
            loader=thirdparty.FileSystemLoader(tempfile.gettempdir()))
        t = e.from_string(d)
        self.assertEqual(t.render(), '')
        self.assertEqual(e.template_parenturi, '/git%20layer.html')

    def test_jinja_concat_extension(self):
        x = 'html'
        f = ['/git.' + x, '/layer.' + x]
        h = sha512(' '.join(f)).hexdigest()[:20]
        d = '{{% concat "{}", "{}" %}}'.format(f[0], f[1])
        e = thirdparty.SandboxedEnvironment(
            extensions=[templating.JinjaConcatExtension],
            loader=thirdparty.FileSystemLoader(tempfile.gettempdir()))
        t = e.from_string(d)
        u = '/' + h + '.' + x
        self.assertEqual(t.render(), u)
        self.assertEqual(e.template_concat, {u: f})

    def test_jinja_markdown_extension(self):
        d = '{% markdown %}# gitlayer{% endmarkdown %}'
        e = thirdparty.SandboxedEnvironment(
            extensions=[templating.JinjaMarkdownExtension],
            loader=thirdparty.FileSystemLoader(tempfile.gettempdir()))
        t = e.from_string(d)
        self.assertEqual(t.render(), '<h1 id="gitlayer">gitlayer</h1>')

    def test_scss_render(self):
        t = '.a { .b { .c { git: layer; } } }'
        self.assertEqual(templating.render_scss(t), '.a .b .c {\n  git: layer; }\n')

    def test_bare_markdown_render(self):
        d = '# gitlayer'
        mdr = templating.MarkdownRenderer(d)
        self.assertEqual(mdr.body, '<h1 id="gitlayer">gitlayer</h1>')

    def test_yaml_markdown_render(self):
        d = '---\nyaml: header\n---\n# gitlayer'
        mdr = templating.MarkdownRenderer(d)
        self.assertEqual(mdr.header, {'yaml': 'header'})
        self.assertEqual(mdr.body, '<h1 id="gitlayer">gitlayer</h1>')

    def test_css_compression(self):
        d = '.a {\n\tgit: layer;\n}\n'
        self.assertEqual(templating.compress_css(d), '.a{git:layer}')

    def test_javascript_comression(self):
        d = 'f = function() {\n\tvar a = 1;\n\tvar b = 2\n\tvar c = 3;\n}\n'
        self.assertEqual(templating.compress_js(d), 'f=function(){var a=1;var b=2;var c=3;};')

    def test_complete_render_jinja(self):
        u = '/gitlayer.html'
        class i(object):
            def __getattr__(self, k):
                return ''
        f = fileobj.GitLayerFile()
        f.do_jinja = True
        f.uri = u
        f.content = '{{ this.uri }}'
        self.assertEqual(templating.render_file(i(), f), u)

    def test_complete_render_scss(self):
        f = fileobj.GitLayerFile()
        f.content = '.a { .b { .c { git: layer; } } }'
        f.do_scss = True
        self.assertEqual(templating.render_file(None, f), '.a .b .c {\n  git: layer; }\n')

    def test_complete_render_markdown(self):
        f = fileobj.GitLayerFile()
        f.content = '# gitlayer'
        f.do_markdown = True
        self.assertEqual(templating.render_file(None, f), '<h1 id="gitlayer">gitlayer</h1>')

# eof