# -*- coding: utf-8 -*-

import sys
from datetime import datetime
from hashlib import sha512

from .errors import GitLayerError
from .thirdparty import (SandboxedEnvironment, JinjaExtension, FileSystemLoader,
    jinjaextwith, jinjaextautoescape, jinjanodes, ScssParser, pytz, markdown,
    UnknownTimeZoneError, csscompress, jscompress, yaml)
from .fileobj import GitLayerFile
from .locales import localeman
from .utils import (soft_parse_date, bytescale, force_bytes,
    file_object_sort, JsonReader, quoteuri)

class JinjaGroupsExtension(JinjaExtension):
    '''
        Extracts metadata stored in {% groups %}{% endgroups %} blocks
        and stores it in the file environment instance. Renders as
        a blank string in templates.
    '''

    tags = set(['groups'])

    def __init__(self, environment):
        super(JinjaGroupsExtension, self).__init__(environment)
        environment.extend(template_groups=set())

    def parse(self, parser):
        lineno = parser.stream.next().lineno
        body = parser.parse_statements(['name:endgroups'], drop_needle=True)
        cb = self.call_method('_parse_groups', [])
        return jinjanodes.CallBlock(cb, [], [], body).set_lineno(lineno)

    def _parse_groups(self, caller):
        groups = set()
        for group in caller().split(','):
            group = group.strip()
            if group:
                groups.add(group)
        self.environment.template_groups = groups | self.environment.template_groups
        return ''

class JinjaTagsExtension(JinjaExtension):
    '''
        Extracts metadata stored in {% tags %}{% endtags %} blocks
        and stores it in the file environment instance. Renders as
        a blank string in templates.
    '''

    tags = set(['tags'])

    def __init__(self, environment):
        super(JinjaTagsExtension, self).__init__(environment)
        environment.extend(template_tags=set())

    def parse(self, parser):
        lineno = parser.stream.next().lineno
        body = parser.parse_statements(['name:endtags'], drop_needle=True)
        cb = self.call_method('_parse_tags', [])
        return jinjanodes.CallBlock(cb, [], [], body).set_lineno(lineno)

    def _parse_tags(self, caller):
        tags = set()
        for tag in caller().split(','):
            tag = tag.strip()
            if tag:
                tags.add(tag)
        self.environment.template_tags = tags | self.environment.template_tags
        return ''

class JinjaDateExtension(JinjaExtension):
    '''
        Extracts metadata stored in a {% date '...' %} statement
        and stores it in the file environment instance. Renders as
        a blank string in templates.
    '''

    tags = set(['date'])

    def __init__(self, environment):
        super(JinjaDateExtension, self).__init__(environment)
        environment.extend(template_date=None)

    def parse(self, parser):
        lineno = parser.stream.next().lineno
        arg = parser.parse_expression()
        arg = getattr(arg, 'value', '')
        # attempt to validate the date
        d = soft_parse_date(arg)
        if d:
            self.environment.template_date = d
        cb = self.call_method('_return_nothing', [])
        return jinjanodes.CallBlock(cb, [], [], []).set_lineno(lineno)

    def _return_nothing(self, caller):
        return ''

class JinjaLocaleExtension(JinjaExtension):
    '''
        Extracts data stored in a {% locale '...' %} statement
        and attempts to set the pages locale.
    '''

    tags = set(['locale'])

    def __init__(self, environment):
        super(JinjaLocaleExtension, self).__init__(environment)
        environment.extend(template_locale='')

    def parse(self, parser):
        lineno = parser.stream.next().lineno
        arg = parser.parse_expression()
        arg = getattr(arg, 'value', '')
        # attempt to validate the locale
        if arg and localeman.switch(arg) and not self.environment.template_locale:
            self.environment.template_locale = arg
        localeman.restore()
        cb = self.call_method('_return_nothing', [])
        return jinjanodes.CallBlock(cb, [], [], []).set_lineno(lineno)

    def _return_nothing(self, caller):
        return ''

class JinjaTimezoneExtension(JinjaExtension):
    '''
        Extracts data stored in a {% timezone '...' %} statement
        and attempts to set the pages timezone.
    '''

    tags = set(['timezone'])

    def __init__(self, environment):
        super(JinjaTimezoneExtension, self).__init__(environment)
        environment.extend(template_timezone=None)

    def parse(self, parser):
        lineno = parser.stream.next().lineno
        arg = parser.parse_expression()
        arg = getattr(arg, 'value', '')
        # attempt to validate the timezone
        try:
            tz = pytz.timezone(arg)
            if not self.environment.template_timezone:
                self.environment.template_timezone = tz
        except UnknownTimeZoneError:
            pass
        cb = self.call_method('_return_nothing', [])
        return jinjanodes.CallBlock(cb, [], [], []).set_lineno(lineno)

    def _return_nothing(self, caller):
        return ''

class JinjaUriExtension(JinjaExtension):
    '''
        Extracts data stored in a {% uri '...' %} statement
        and attempts to set the pages URI.
    '''

    tags = set(['uri'])

    def __init__(self, environment):
        super(JinjaUriExtension, self).__init__(environment)
        environment.extend(template_uri=None)

    def parse(self, parser):
        lineno = parser.stream.next().lineno
        arg = parser.parse_expression()
        arg = getattr(arg, 'value', '')
        self.environment.template_uri = quoteuri(arg)
        cb = self.call_method('_return_nothing', [])
        return jinjanodes.CallBlock(cb, [], [], []).set_lineno(lineno)

    def _return_nothing(self, caller):
        return ''

class JinjaConcatExtension(JinjaExtension):
    '''
        Extracts data stored in a {% concat '...' %} statement
        and returns a URI to a concatenated resource.
    '''

    tags = set(['concat'])

    def __init__(self, environment):
        super(JinjaConcatExtension, self).__init__(environment)
        environment.extend(template_concat={})

    def parse(self, parser):
        lineno = parser.stream.next().lineno
        args = [parser.parse_expression()]
        if parser.stream.skip_if('comma'):
            args.append(parser.parse_expression())
        else:
            args.append(jinjanodes.Const(None))
        concat_files = []
        for arg in args:
            concat_files.append(quoteuri(getattr(arg, 'value', '')))
        concat_files = sorted(concat_files)
        if concat_files:
            c_ext = concat_files[-1].split('.')[-1]
            c_hash = sha512(' '.join(concat_files)).hexdigest()
            concat_uri = '/' + c_hash[:20] + '.' + c_ext
            self.environment.template_concat[concat_uri] = concat_files
        cb = self.call_method('_return_concat_uri', [jinjanodes.Const(concat_uri)])
        return jinjanodes.CallBlock(cb, [], [], []).set_lineno(lineno)

    def _return_concat_uri(self, *a, **k):
        uri = a[0]
        return uri

class JinjaParentExtension(JinjaExtension):
    '''
        Extracts data stored in a {% parent '...' %} statements
        and returns a URI to a concatenated resource.
    '''

    tags = set(['parent'])

    def __init__(self, environment):
        super(JinjaParentExtension, self).__init__(environment)
        environment.extend(template_parenturi=None)

    def parse(self, parser):
        lineno = parser.stream.next().lineno
        arg = parser.parse_expression()
        arg = getattr(arg, 'value', '')
        self.environment.template_parenturi = quoteuri(arg)
        cb = self.call_method('_return_nothing', [])
        return jinjanodes.CallBlock(cb, [], [], []).set_lineno(lineno)

    def _return_nothing(self, caller):
        return ''

class JinjaMarkdownExtension(JinjaExtension):
    '''
        Extracts data stored in a {% markdown %}{% endmarkdown %} blocks
        and renders it through the markdown processor.
    '''

    tags = set(['markdown'])

    def parse(self, parser):
        lineno = parser.stream.next().lineno
        body = parser.parse_statements(['name:endmarkdown'], drop_needle=True)
        cb = self.call_method('_parse_markdown', [])
        return jinjanodes.CallBlock(cb, [], [], body).set_lineno(lineno)

    def _parse_markdown(self, caller):
        return MarkdownRenderer(caller()).body

def datetimeformat_filter(value, format='%Y-%m-%d %H:%M:%S'):
    format = str(format)
    if type(value) != datetime:
        return ''
    try:
        return value.strftime(format)
    except (TypeError, ValueError) as e:
        return ''

def bytestokilo_filter(value, rounding=2):
    return bytescale(value, rounding, depth=1)

def bytestomega_filter(value, rounding=2):
    return bytescale(value, rounding, depth=2)

def jinja_render(template_string='', context_dict={}, env=None):
    template = env.from_string(template_string)
    return template.render(**context_dict)

def jinja_env(path=''):
    loader = FileSystemLoader(path)
    env = SandboxedEnvironment(extensions=[
            JinjaGroupsExtension,
            JinjaTagsExtension,
            JinjaDateExtension,
            JinjaLocaleExtension,
            JinjaTimezoneExtension,
            JinjaMarkdownExtension,
            JinjaUriExtension,
            JinjaConcatExtension,
            JinjaParentExtension,
            jinjaextwith,
            jinjaextautoescape],
        trim_blocks=True,
        loader=loader)
    env.filters['dateformat'] = datetimeformat_filter
    env.filters['bytestokilo'] = bytestokilo_filter
    env.filters['bytestomega'] = bytestomega_filter
    return env

def blank_context(f):
    '''
        Returns a context that makes the main render but is blank, this is
        to stop initial metadata collection generating rendering errors.
    '''
    return {
        'groups': [], 'tags': [], 'files': [], 'date': None,
        'pathmatch': lambda *x, **y: [], 'this': f,
        'tagmatch': lambda *x, **y: [], 'groupmatch': lambda *x, **y: [],
        'now': datetime.now(), 'utcnow': datetime.utcnow(),
        'byuri': lambda x: GitLayerFile(), 'pages': [], 'posts': [],
        'postsreversed': [], 'datetree': lambda *x: {},
        'filesort': file_object_sort, 'datematch': lambda *x, **y: [],
        'nextpost': lambda *x, **y: None, 'previouspost': lambda *x, **y: None,
        'jsonfile': lambda *x, **y: None,
    }

def renderer_context(index, f):
    '''
        Return the extended context for all final jinja2 rendering which provides
        the groups, tags, blocks and other extra variables to the template.
    '''
    return {
        # local file request
        'this': f,
        # global lists of files
        'groups': index.all_groups,
        'tags': index.all_tags,
        'files': index.all_files,
        'pages': index.all_pages,
        'posts': index.all_posts,
        # interactive searches
        'pathmatch': index.file_match_by_uri,
        'tagmatch': index.file_match_by_tags,
        'groupmatch': index.file_match_by_groups,
        'byuri': index.get_file,
        'datetree': index.posts_into_datetree,
        'datematch': index.posts_by_daterange,
        'nextpost': index.next_post,
        'previouspost': index.prev_post,
        # local current time
        'now': datetime.now(),
        'utcnow': datetime.utcnow(),
        # python functions exposed to templates
        'filesort': file_object_sort,
        'jsonfile': JsonReader(index.path).readjson,
    }

scss_compiler = ScssParser()
def render_scss(s):
    return scss_compiler.compile(force_bytes(s))

class MarkdownRenderer(object):
    '''
        Wrapper on markdown rendering to extract an optional yaml-formatted header.
    '''

    HEADER_MARKER = '---'

    def __init__(self, s, silent=False):
        self.raw_body = s.strip().replace('\r', '')
        self.raw_body = force_bytes(self.raw_body)
        self.silent = silent
        self.raw_header = ''
        self.parsed_header = {}
        self.check_header()
        self.rendered_body = self.render_body()

    def check_header(self):
        header = ''
        h_s = self.raw_body.find(self.HEADER_MARKER)
        if h_s == 0:
            h_e = self.raw_body.find(self.HEADER_MARKER, 1)
            header = self.raw_body[h_s + len(self.HEADER_MARKER):h_e].strip()
        if header:
            self.raw_body = self.raw_body[h_e + len(self.HEADER_MARKER):].strip()
            self.parsed_header = self._parse_yaml(header)

    def _parse_yaml(self, s):
        try:
            return yaml.load(s)
        except Exception as e:
            if self.silent:
                return {}
            else:
                raise GitLayerError('error in markdown yaml header: ' + e.message)

    def render_body(self):
        self.raw_body = self.raw_body.decode('utf-8')
        return markdown(self.raw_body, extensions=['extra', 'abbr', 'attr_list',
            'def_list', 'fenced_code', 'footnotes', 'tables', 'smart_strong',
            'codehilite', 'headerid', 'sane_lists', 'smarty', 'toc',
            'wikilinks(base_url=/,end_url=.html)'])

    @property
    def header(self):
        return self.parsed_header

    @property
    def body(self):
        return self.rendered_body

def compress_css(s):
    return csscompress(force_bytes(s))

def compress_js(s):
    return jscompress(force_bytes(s))

def render_file(index, f, callback=None, errback=None):
    '''
        Attempt to render a file object passing it through filters and renderers if
        required. If the optional callback is supplied the data will be passed to it
        rather than returned.
    '''
    output = ''
    data = f.content if f.dynamic else index.readfile(index.full_path(f))
    if f.do_jinja:
        # templates need rendering through jinja2
        try:
            if f.locale_changed:
                localeman.switch(f.locale)
            html = jinja_render(
                data,
                renderer_context(index, f),
                jinja_env(index.path))
            localeman.restore()
        except Exception as e:
            if errback:
                exc_type, exc_value, exc_traceback = sys.exc_info()
                if getattr(e, 'filename', None) == None:
                    e.filename = f.uri
                errback((e, exc_traceback))
            else:
                raise
            return False
        output = html
    elif f.do_scss:
        # scss files are rendered through pyScss
        try:
            css = render_scss(data)
        except Exception as e:
            if errback:
                exc_type, exc_value, exc_traceback = sys.exc_info()
                if getattr(e, 'filename', None) == None:
                    e.filename = f.uri
                errback((e, exc_traceback))
            else:
                raise
            return False
        output = css
    elif f.do_markdown:
        # markdown files with optional YAML headers are rendered through markdown
        try:
            html = MarkdownRenderer(data).body
        except Exception as e:
            if errback:
                exc_type, exc_value, exc_traceback = sys.exc_info()
                if getattr(e, 'filename', None) == None:
                    e.filename = f.uri
                errback((e, exc_traceback))
            else:
                raise
            return False
        output = html
    elif f.dynamic:
        # dynamic files are stored in the file objects and not read off the disk
        output = f.content
    else:
        # some other file type, return it verbatim
        output = data
    # post-rendering compression
    if f.do_minify:
        if f.mime == 'application/javascript':
            try:
                output = compress_js(output)
            except Exception as e:
                if errback:
                    exc_type, exc_value, exc_traceback = sys.exc_info()
                    e.filename = f.uri
                    errback((e, exc_traceback))
                else:
                    raise
        elif f.mime == 'text/css':
            try:
                output = compress_css(output)
            except Exception as e:
                if errback:
                    exc_type, exc_value, exc_traceback = sys.exc_info()
                    e.filename = f.uri
                    errback((e, exc_traceback))
                else:
                    raise
    # callback or return
    if callback:
        return callback(force_bytes(output))
    else:
        return output

# eof
