# -*- coding: utf-8 -*-

import os
import sys
import getopt
from shutil import rmtree

from . import log, __version__ as version
from .fileindex import GitLayerIndex
from .handler import GitLayerRequestHandler
from .server import HTTPServer
from .build import BuildSite

DEFAULT_HOST = '127.0.0.1'
DEFAULT_PORT = 8000

def usage():
    print('GitLayer local server arguments:')
    print(' -h --help     This help message')
    print(' -v --version  Prints the GitLayer installation version number')
    print(' -b --bind     IP or host to bind the server to, defaults to 127.0.0.1')
    print(' -p --port     Port number to listen on, defaults to 8000')
    print(' -r --root     Directory to serve, defaults to the current working dir')
    print(' -o --out      Path to a directory to save a rendered copy of the site')
    print(' -d --delete   When combined with -o the output directory is deleted first')

def run():
    try:
        opts, args = getopt.getopt(sys.argv[1:], 'hvb:p:r:o:d',
            ['help', 'version', 'bind', 'port', 'root', 'out', 'delete'])
    except getopt.GetoptError as err:
        log.error(err)
        usage()
        sys.exit(2)
    host = DEFAULT_HOST
    port = DEFAULT_PORT
    target_dir = os.getcwd()
    build_dir = None
    delete_dir = False
    for opt, arg in opts:
        if opt in ('-h', '--help'):
            usage()
            sys.exit()
        elif opt in ('-v', '--version'):
            sys.stdout.write('GitLayer {}\n'.format(version))
            sys.exit()
        elif opt in ('-b', '--bind'):
            host = arg
        elif opt in ('-p', '--port'):
            port = arg
        elif opt in ('-r', '--root'):
            target_dir = arg
        elif opt in ('-o', '--out'):
            build_dir = arg
        elif opt in ('-d', '--delete'):
            delete_dir = True
    if len(args) > 0:
        # allow for just 0.0.0.0:0000 syntax
        parts = args[0].split(':')
        if len(parts) == 1:
            if parts[0].find('.') > 0:
                _host, _port = parts[0], DEFAULT_PORT
            else:
                _host, _port = DEFAULT_HOST, parts[0]
        elif len(parts) == 2:
            _host, _port = parts
        host = _host if _host != host else host
        port = _port if _port != port else port
    if build_dir:
        # build the site then exit
        build_site(target_dir, build_dir, delete_dir)
    else:
        # start up a development server
        serve_site(target_dir, host, port)

def build_site(target_dir, output_dir, delete_dir=False):
    target_dir = os.path.abspath(os.path.expandvars(os.path.expanduser(target_dir)))
    output_dir = os.path.abspath(os.path.expandvars(os.path.expanduser(output_dir)))
    log.info('building site from {} into {}'.format(target_dir, output_dir))
    if not os.path.isdir(target_dir):
        log.error('site directory does not exist: {}'.format(target_dir))
        sys.exit(2)
    if os.path.isdir(output_dir):
        if delete_dir:
            log.info('You have requested to DELETE all files in the output directory before rendering into it')
            log.info('This will remove all existing files in the directory: {}'.format(output_dir))
            log.info('Do you confirm this? y/n')
            yes = ('yes', 'y', 'ye')
            no = ('no', 'n')
            while True:
                choice = raw_input().lower()
                if choice in yes:
                    log.info('Got confirmation... deleting all files in output directory')
                    rmtree(output_dir)
                    break
                elif choice in no:
                    log.error('Confirmation declined, aborting rendering...')
                    sys.exit(2)
                else:
                    log.info('Please respond with \'yes\' or \'no\'')
        else:
            log.error('output directory already exists (use -d to automatically delete): {}'.format(output_dir))
            sys.exit(2)
    builder = BuildSite(target_dir, output_dir)
    builder.build()
    log.info('build complete')

def serve_site(target_dir, host, port):
    log.info('attempting to serve directory {}'
        .format(target_dir, host, port))
    if not os.path.isdir(target_dir):
        log.error('site directory does not exist: {}'.format(target_dir))
        sys.exit(2)
    try:
        port = int(port)
    except ValueError:
        log.error('optional port must be a number, got: {}'.format(port))
        sys.exit(2)
    if port < 1024 or port > 65535:
        log.error('port must be >1024 and <65535, got: {}'.format(port))
        sys.exit(2)
    GitLayerRequestHandler.index = GitLayerIndex(target_dir)
    http_server = HTTPServer((host, port), GitLayerRequestHandler)
    log.info('GitLayer development server listening on: http://{}:{}/'
        .format(host, port))
    log.info('Use control+c to stop')
    try:
        http_server.serve_forever()
    except KeyboardInterrupt:
        log.info('caught control+c, shutting down...')
        http_server.socket.close()

# eof
