# -*- coding: utf-8 -*-

from datetime import datetime

from .locales import localeman

class GitLayerFile(object):
    '''
        Represents a single file in the web root. This is available to
        the template.
    '''

    def __init__(self):
        self.file = ''              # file name
        self.dir = ''               # bare directory name (from URI)
        self.uri = ''               # complete URI of resource, may be remapped
        self.original_uri = ''      # complete original URI of the resource
        self.remapped_uri = ''      # complete remapped URI of the resource
        self.parent_uri = ''        # the URI of the parent object
        self.parent = None          # the parent file object of this file
        self.name = ''              # bare file name minus the extension
        self.nameparts = []         # list of self.name split by hyphens
        self.ext = ''               # file extension
        self.size = 0               # file size in bytes
        self.large = False          # True if file is > MAX_RENDER_SIZE
        self.mime = ''              # file mime type
        self.do_jinja = False       # file needs to be rendered through jinja2
        self.do_scss = False        # file needs to be rendered through pyScss
        self.do_concatenation = False # file needs concatention of the related files
        self.do_minify = False      # file needs minification (css and js only)
        self.do_markdown = False    # file needs yaml and markdown rendering
        self.is_error401 = False    # file is the error 401 file
        self.is_error404 = False    # file is the error 404 file
        self.is_alias = False       # file is an alias of another file (e.g. /)
        self.is_remap = False       # file has had its URI remapped
        self.is_base = False        # file is a template (name starts with _)
        self.is_html = False        # file is an HTML page and is not an error page
        self.locale = localeman.default_locale
        self.locale_changed = False
        self.timezone = None
        self.date = None            # datetime.datetime instance if set in template
        self.utcdate = None         # same as above but always with tzinfo=<UTC>
        self._groups = set()        # set of arbitrary groups for file
        self._tags = set()          # set of arbitrary tags for file
        self._modtime = 0           # timestamp when the file was last modified
        self.blocks = GitLayerBlocks()
        self.content = ''           # replaces local file content if set
        self._related_files = []    # any files related to this one

    def __str__(self):
        return '<GitLayerFile {}>'.format(self.uri)

    @property
    def groups(self):
        return sorted(list(self._groups))

    @property
    def tags(self):
        return sorted(list(self._tags))

    @property
    def dynamic(self):
        return self.content != ''

    def localise_date(self, dt=None):
        if not self.timezone or type(dt) != datetime:
            return dt
        return self.timezone.localize(dt)

    @property
    def do_something(self):
        # returns a str of the rendering required if any post-processing is needed
        if self.do_jinja:
            return 'jinja2'
        elif self.do_scss:
            return 'scss'
        elif self.do_markdown:
            return 'markdown'
        elif self.dynamic:
            return 'dynamic'
        return False

    @property
    def parents(self):
        # recurse backwards through each parent to the root of the tree
        parent = self.parent
        while parent:
            yield parent
            parent = getattr(parent, 'parent')

class GitLayerBlocks(object):
    '''
        Basic object to allow pretty access to block values.
    '''

    def __init__(self):
        self._blocks = {}

    def __str__(self):
        k = ','.join(self._blocks.keys())
        return '<GitLayerBlock [{}]>'.format(k if k else 'empty')

    def __get__(self, name):
        return self._blocks.get(name, '')

    def __set__(self, name, value):
        self._blocks[name] = value

# eof
