# -*- coding: utf-8 -*-

import os
import unittest
import tempfile
import shutil

from gitlayer import (fileindex, errors)

class TestFileIndex(unittest.TestCase):

    def setUp(self):
        self.data = os.urandom(32)
        self.fakesite = tempfile.mkdtemp()
        os.mkdir(os.path.join(self.fakesite, 'b'))
        os.mkdir(os.path.join(self.fakesite, 'c'))
        os.mkdir(os.path.join(self.fakesite, 'd'))
        self.fakefiles = []
        f = os.path.join(self.fakesite, 'a.html')
        open(f, 'w').write('{% groups %}a, b{% endgroups %}{% tags %}c, d{% endtags %}{% date "1970-01-01" %}{{ this.uri }}')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'b.html')
        open(f, 'w').write('{% date "1970-01-02" %}')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'c.html')
        open(f, 'w').write('{% concat "/a.html", "/p.html" %}')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, '401error.html')
        open(f, 'w').write('<p>error 401</p>')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, '404error.html')
        open(f, 'w').write('<p>error 404</p>')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, '_t.html')
        open(f, 'w').write('<p>template</p>')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'b/b.scss.css')
        open(f, 'w').write('.a { .b { .c { git: layer; } } }')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'c/c.md')
        open(f, 'w').write('# gitlayer')
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'd/d.jpg')
        open(f, 'w').write(self.data)
        self.fakefiles.append(f)
        f = os.path.join(self.fakesite, 'm.md')
        open(f, 'w').write('---\nuri: /m.html\n---\n#content')
        self.fakefiles.append(os.path.join(self.fakesite, 'm.html'))
        self.sitemapfile = '/sitemap.xml'
        self.concatfile = '/20b147f91fd109b1d473.html'
        self.defaultstate = [self.sitemapfile, self.concatfile]
        self.fakeindex = fileindex.GitLayerIndex(self.fakesite)

    def test_fileindex_dircheck(self):
        d = os.urandom(32)
        with self.assertRaises(errors.GitLayerError):
            fileindex.GitLayerIndex(path='/gitlayer/' + d.encode('hex'))

    def test_fileindex_scan(self):
        fake_uris = self.defaultstate
        for f in self.fakefiles:
            fake_uris.append(f[len(self.fakesite):])
        index_uris = []
        for f in self.fakeindex.all_files_and_errors:
            index_uris.append(f.uri)
        self.assertEqual(sorted(fake_uris), sorted(index_uris))

    def test_fileindex_errorpages(self):
        self.assertEqual(self.fakeindex.error401.uri, '/401error.html')
        self.assertEqual(self.fakeindex.error404.uri, '/404error.html')

    def test_fileindex_geturi(self):
        self.assertEqual(self.fakeindex.get_file(self.sitemapfile).uri, self.sitemapfile)

    def test_fileindex_fullpath(self):
        u = os.path.join(self.fakesite, self.sitemapfile[1:])
        f = self.fakeindex.get_file(self.sitemapfile)
        self.assertEqual(self.fakeindex.full_path(f), u)

    def test_fileindex_metadata(self):
        f = self.fakeindex.get_file('/a.html')
        self.assertEqual(f.groups, ['a', 'b'])
        self.assertEqual(f.tags, ['c', 'd'])
        f = self.fakeindex.get_file('/b.html')
        self.assertEqual(f.date.year, 1970)
        self.assertEqual(f.date.month, 1)
        self.assertEqual(f.date.day, 2)

    def test_fileindex_mime(self):
        self.assertEqual(self.fakeindex._get_mime_type('atom'), 'application/atom+xml')
        self.assertEqual(self.fakeindex._get_mime_type('crt'), 'application/x-x509-ca-cert')
        self.assertEqual(self.fakeindex._get_mime_type(None), 'text/plain')
        self.assertEqual(self.fakeindex._get_mime_type(''), self.fakeindex.DEFAULT_MIME)
        f = self.fakeindex.get_file('/a.html')
        self.assertEqual(f.mime, 'text/html')
        f = self.fakeindex.get_file('/b/b.scss.css')
        self.assertEqual(f.mime, 'text/css')
        f = self.fakeindex.get_file('/c/c.md')
        self.assertEqual(f.mime, 'text/html')
        f = self.fakeindex.get_file('/d/d.jpg')
        self.assertEqual(f.mime, 'image/jpeg')

    def test_fileindex_files(self):
        fake_uris = self.defaultstate
        for f in self.fakefiles:
            if not f.endswith('error.html'):
                fake_uris.append(f[len(self.fakesite):])
        index_uris = []
        for f in self.fakeindex.all_files:
            index_uris.append(f.uri)
        self.assertEqual(sorted(fake_uris), sorted(index_uris))

    def test_fileindex_modtime(self):
        f = self.fakeindex.get_file('/a.html')
        d = self.fakeindex.file_utc_modtime(f)
        self.assertEqual(d.year, 1970)
        self.assertEqual(d.month, 1)
        self.assertEqual(d.day, 1)

    def test_fileindex_posts(self):
        fake_uris = ['/a.html', '/b.html']
        index_uris = []
        for f in self.fakeindex.all_posts:
            index_uris.append(f.uri)
        self.assertEqual(sorted(fake_uris), sorted(index_uris))

    def test_fileindex_groups(self):
        self.assertEqual(self.fakeindex.all_groups, ['a', 'b'])

    def test_fileindex_tags(self):
        self.assertEqual(self.fakeindex.all_tags, ['c', 'd'])

    def test_fileindex_bygroups(self):
        f = [f.uri for f in self.fakeindex.file_match_by_groups(['a', 'b'])]
        self.assertEqual(f, ['/a.html'])

    def test_fileindex_bytags(self):
        f = [f.uri for f in self.fakeindex.file_match_by_tags(['c', 'd'])]
        self.assertEqual(f, ['/a.html'])

    def test_fileindex_byuri(self):
        f = [f.uri for f in self.fakeindex.file_match_by_uri('/a.*')]
        self.assertEqual(f, ['/a.html'])

    def test_fileindex_datetree(self):
        f = [self.fakeindex.get_file('/a.html').uri, self.fakeindex.get_file('/b.html').uri]
        s = []
        for d1,v1 in self.fakeindex.posts_into_datetree().iteritems():
            for d2,v2 in v1.iteritems():
                for d3,v3 in v2.iteritems():
                    for x in v3:
                        s.append(x.uri)
        self.assertEqual(sorted(s), f)

    def test_fileindex_postrage(self):
        f = [self.fakeindex.get_file('/a.html').uri, self.fakeindex.get_file('/b.html').uri]
        s = self.fakeindex.posts_by_daterange(date_from='1969-01-01', date_to='1971-01-01')
        self.assertEqual(sorted([x.uri for x in s]), f)

    def text_fileindex_nextpost(self):
        f1 = self.fakeindex.get_file('/a.html')
        f2 = self.fakeindex.get_file('/b.html')
        self.assertEqual(self.fakeindex.next_post(f1), f2)

    def text_fileindex_prevpost(self):
        f1 = self.fakeindex.get_file('/a.html')
        f2 = self.fakeindex.get_file('/b.html')
        self.assertEqual(self.fakeindex.next_post(f2), f1)

    def tearDown(self):
        shutil.rmtree(self.fakesite)

# eof