#    GMail-Notifier - A cooler looking gmail notifier for Ubuntu
#    Copyright (C) 2009 Uday K Verma
#    Copyright (C) 2009 Ritesh Nadhani
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.


import sys
import logging
import gmailnotifier
import gobject, gtk

from gmailnotifier import mailchecker
from gmailnotifier import statusiconmanager
from gmailnotifier import notificationmanager
from gmailnotifier import config
from gmailnotifier import prefs
from gmailnotifier import launcher
from gmailnotifier import updatechecker

class Notifier (gobject.GObject):
    __gsignals__ = { 'request-quit' : (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, ()), }
    
    def __init__ (self):
        gobject.GObject.__init__ (self)
        
        gobject.set_application_name ("GMail-Notifier")
        
        self.uc = updatechecker.UpdateChecker ()
        self.sm = statusiconmanager.StatusIconManager (self.uc)
        self.nm = notificationmanager.NotificationManager (self.sm)
        self.mc = mailchecker.MailChecker (config.get_config ("username"),
                                           config.get_password (config.get_config ("username")))

        self._connectStatusIconHandlers ()
        self._connectMailCheckerHandlers ()
        
    def new_mail_callback (self, sender, messages):
        self.sm.updateForMessages (messages)
        self.nm.notifyMessages (messages)
        
    def failed_mail_callback (self, sender):
        self.sm.updateForFailed ()
        self.nm.updateForFailed ()
        
    def refresh_callback (self, sender):
        self.nm.clearCache ()
        self.mc.checkMail ()
        
    def quit_callback (self, sender):
        self.nm.close ()
        self.emit ("request-quit")
        
    def mail_checker(self):
        self.mc.checkMail ()
        
        # we always want this timer to die, because we establish a new one every
        # time we're called.  This is because we want the new refresh-delay
        # to take effect
        #
        gobject.timeout_add (int(config.get_config('refresh-interval')), self.mail_checker)
        return False
    
    def on_creds_saved (self, sender):
        self.mc = mailchecker.MailChecker (config.get_config("username"),
                                  config.get_password (config.get_config ("username")))
        self._connectMailCheckerHandlers ()
        
        self.nm.stop ()
    
    def prefs_callback (self, sender):
        p = prefs.PrefWindow ()
        p.connect ('credentials-saved', self.on_creds_saved)
    
    def notification_toggle_callback(self, sender, enabled):
        if enabled:
            config.set_config ("disable-notifications", "1")
        else:
            config.set_config ("disable-notifications", "0")
        
    def inbox_callback (self, sender):
        launcher.browser ("http://gmail.google.com/mail")
        
    def mail_callback (self, sender, link):
        launcher.browser (link)
        
    def activate_callback (self, sender):
        self.nm.stop ()
        
    def start (self):
        if not config.has_configuration ():
            config.setup_default_configuration ()
            self.prefs_callback (None)
    
        # check mail once before starting up (this will also setup the timer
        # for the next time around)
        self.mail_checker ()
        
    def _connectMailCheckerHandlers (self):
        # connect all mail checker notifications
        self.mc.connect ('new-messages', self.new_mail_callback)
        self.mc.connect ('failed-messages', self.failed_mail_callback)
    
    
    def _connectStatusIconHandlers (self):
        # connect all status icon manager notifications
        self.sm.connect ('request-refresh', self.refresh_callback)
        self.sm.connect ('request-quit', self.quit_callback)
        self.sm.connect ('request-notification-disable', self.notification_toggle_callback)
        self.sm.connect ('request-prefs', self.prefs_callback)
        self.sm.connect ('request-inbox', self.inbox_callback)
        self.sm.connect ('request-mail', self.mail_callback)
        self.sm.connect ('request-activate', self.activate_callback)
