"""@package graphlab.connect.aws.ec2

This module makes it easy to have a GraphLab server running in EC2.
"""

import json
import logging
import os
import time
import urllib2

import boto.ec2
from boto.exception import EC2ResponseError

import graphlab.product_key
import graphlab.connect.server as glserver
import graphlab.connect.main as glconnect
from graphlab.connect.main import __catch_and_log__

from graphlab.cython.ipc import PyCommClient as Client

from graphlab_util.config import DEFAULT_CONFIG as config
from graphlab.connect import _get_metric_tracker

GRAPHLAB_PORT_MIN_NUM = 9000
GRAPHLAB_PORT_MAX_NUM = 9002
JSON_BLOB_PATH_FORMAT = '/gl_api/api/v1/cloud/aws/instances/%s?client_version=%s'
REGION = 'us-west-2'
SECURITY_GROUP_NAME = 'GraphLab'

__server__ = None

# Swallow all boto logging, except critical issues
logging.getLogger('boto').setLevel(logging.CRITICAL)

__LOGGER__ = logging.getLogger(__name__)

class _EC2(glserver.RemoteServer):
    """
    The class which manages the connection information for a unity server
    running on an EC2 instance.
    """
    def __init__(self, instance_type, CIDR_rule, auth_token, product_key):
        self.logger = __LOGGER__
        self.region = REGION
        self.auth_token = auth_token
        
        _get_metric_tracker().track(('ec2.instance_type.%s' % instance_type), value=1)

        try:
            conn = boto.ec2.connect_to_region(self.region)
            self.logger.debug("Starting up a %s EC2 instance in %s. You will be responsible for the cost of this "
                              "instance." % (instance_type, self.region))

            # Does the necessary security group already exist, for this user?
            security_group = None
            for sg in conn.get_all_security_groups():
                if(SECURITY_GROUP_NAME == sg.name):
                    security_group = sg
                    break

        except boto.exception.EC2ResponseError as e:
            self.logger.debug('EC2 Response Error: %s' % e)
            raise Exception('EC2 response error. Please verify your AWS credentials. To configure your AWS credentials use graphlab.aws.set_credentials or set the AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY environment variables')
        except boto.exception.NoAuthHandlerFound as e:
            raise Exception('AWS configuration not found. Please configure your AWS credentials using graphlab.aws.set_credentials or by setting AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY environment variables')

        # Do we need to create a security group?
        if security_group is None:
            security_group = conn.create_security_group(SECURITY_GROUP_NAME,
                                                        'Only open ports needed for GraphLab Server')

        # even if the group exists, just add the GraphLab Server required rules
        try:
            security_group.authorize('tcp', GRAPHLAB_PORT_MIN_NUM, GRAPHLAB_PORT_MAX_NUM, CIDR_rule)
        except EC2ResponseError as e:
            if not "<Code>InvalidPermission.Duplicate</Code>" in str(e):
                raise

        if (config.mode == 'UNIT' and 'GRAPHLAB_TEST_AMI_ID' in os.environ and 'GRAPHLAB_TEST_ENGINE_URL' in os.environ):
            # unit-test mode, don't involve webservice to retrieve AMI, instead use environment variables
            ami_id = os.environ['GRAPHLAB_TEST_AMI_ID']
            engine_url = os.environ['GRAPHLAB_TEST_ENGINE_URL']
            self.logger.info("UNIT mode, using AMI: '%s' and engine url: '%s' when launching EC2 instance." % (ami_id, engine_url))
            json_blob = json.loads('{}')
            json_blob['ami_id'] = ami_id
            json_blob['engine_url'] = engine_url
            json_blob['os_url'] = ''
        else:
            # Get the info to start a EC2 from the GraphLab Server
            json_blob_path = JSON_BLOB_PATH_FORMAT % (instance_type, config.version)
            json_blob_url = config.graphlab_server + json_blob_path

            try:
                graphlab_server_response = urllib2.urlopen(json_blob_url)
                json_blob = json.loads(graphlab_server_response.read())
            except:
                raise Exception("Unable to successfully retrieve correct EC2 image to launch for this version. This could be a temporary problem. Please try again in a few minutes. If the problem persists please contact support@graphlab.com")

        self.logger.debug("web service return: %s" % json_blob)

        if json_blob['ami_id'] is None:
            raise Exception("Unable to successfully retrieve correct EC2 image to launch. Please try again later or contact support@graphlab.com. Error received:'%s'", json_blob['message'])

        ami_id = json_blob['ami_id']

        if self.auth_token is not None:
            json_blob['auth_token'] = self.auth_token
        else:
            json_blob['auth_token'] = ''

        # set product key in json_blob
        if product_key is None:
            product_key = ''
        json_blob['product_key'] = product_key

        run_instances_args =  {
            'security_group_ids' : [ SECURITY_GROUP_NAME ],
            'user_data' : json.dumps(json_blob),
            'instance_type' : instance_type
        }

        if (config.mode == 'UNIT' and 'GRAPHLAB_TEST_EC2_KEYPAIR' in os.environ):
            keypair = os.environ['GRAPHLAB_TEST_EC2_KEYPAIR']
            self.logger.info("UNIT mode, using keypair: '%s' when launching EC2 instance" % (keypair))
            run_instances_args['key_name'] = keypair

        # Launch the EC2 instance
        response = conn.run_instances(ami_id, **run_instances_args)

        if(not(len(response.instances) == 1)):
            raise Exception('Could not successfully start EC2 instance.')

        instance = response.instances[0]

        # If auth_token is not set by the input argument, the ec2 instance
        # will use its instance id as the default auth_token, so we reset
        # self.auth_token after the instance is running.
        if not self.auth_token:
            self.auth_token = instance.id

        self.logger.info("Launching an %s instance in the %s availability zone, with id: %s. You will be responsible for the cost of this instance." % (instance.instance_type, instance.placement, instance.id))

        # Wait for the host to start up
        while not instance.update() == 'pending':
            time.sleep(1)
        while instance.update() == 'pending':
            time.sleep(1)

        if instance.state == 'running':
            self.instance_id = instance.id
            server_addr = 'tcp://' + instance.ip_address + ':' + str(GRAPHLAB_PORT_MIN_NUM)
            self.logger.debug("EC2 instance is now running, attempting to connect to engine.")
            super(self.__class__, self).__init__(server_addr, auth_token=self.auth_token)
        else:   # Error case
            self.logger.error('Could not startup EC2 instance.')
            if(hasattr(instance, 'id')):
                self.stop()

    def get_auth_token(self):
        return self.auth_token

    def stop(self):
        _get_metric_tracker().flush()
        self.logger.info('Terminating EC2 instance, id: %s.' % (self.instance_id))
        try:
            conn = boto.ec2.connect_to_region(self.region)
            response = conn.terminate_instances(instance_ids=[self.instance_id])
            if(not(len(response) == 1)):
                self.logger.error('Invalid response from EC2. Unable to terminate host. Please terminate '
                                'using the AWS EC2 Console.')
                return
            if response[0].update() == 'shutting-down':
                while response[0].update() == 'shutting-down':
                    time.sleep(1)
            else:
                self.logger.error('Invalid response from EC2. Unable to terminate host. Please terminate '
                                'using the AWS EC2 Console.')
        except:
            self.logger.error('Invalid response from EC2. Unable to terminate host. Please terminate '
                              'using the AWS EC2 Console.')



def get_credentials():
    """
    Returns the AWS credential pairs stored in the environment variables:
    AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY.
    """
    if (not 'AWS_ACCESS_KEY_ID' in os.environ):
        raise KeyError('No access key found. Please set the environment variable AWS_ACCESS_KEY_ID, or using graphlab.aws.set_credential()')
    if (not 'AWS_SECRET_ACCESS_KEY' in os.environ):
        raise KeyError('No secret key found. Please set the environment variable AWS_SECRET_ACCESS_KEY, or using graphlab.aws.set_credential()')
    return (os.environ['AWS_ACCESS_KEY_ID'], os.environ['AWS_SECRET_ACCESS_KEY'])


def launch_EC2(instance_type='m3.xlarge', CIDR_rule='0.0.0.0/0', auth_token=None):
    """
    Launches an EC2 instance of the specified type. It is expected that AWS credentials are already set
    as environment variables (see :py:func:`~.graphlab.aws.set_aws_credential()`), or configuration is provided in the Boto config file.

    All EC2 instances are currently launched in the US-WEST-2 (Oregon) AWS region.

    Parameters
    ----------
    instance_type : string, optional
        The EC2 instance type to launch, default is m3.xlarge. We support all instance types
        except for 'm1.small' and 't1.micro'. For a list of instance_types, please
        refer to `here <http://aws.amazon.com/ec2/instance-types/#Instance_Types>`_.

    CIDR_rule : string, optional
        The Classless Inter-Domain Routing rule to use for the instance. Useful for restricting the IP
        Address Range for a client.

    auth_token : string, optional
        The Authentication Token to be used by the instance.

    """
    # Check existing connection
    if glconnect.is_connected():
        __LOGGER__.warning('You have GraphLab objects instantiated on the current server. If you want to \n'
                           'launch a new server, you must first stop the current server connection by running: \n'
                           '\'graphlab.connect.main.stop()\'. Be warned: you will lose these instantiated objects.')
        return

    global __server__
    assert __server__ is None
    assert not glconnect.is_connected()

    # get and validate the product / registration key
    try:
        product_key = graphlab.product_key.get_product_key()
    except KeyError as k:
        __LOGGER__.error(k.message)
        return

    if (not graphlab.product_key.is_product_key_valid(product_key)):
        __LOGGER__.error("Product Key validation failed, please confirm your product key is correct. "
                         "If you believe this key to be valid, please contact support@graphlab.com")
        return

    # By default the EC2 instance will use its instance id as the auth_token.
    try:
        __server__ = _EC2(instance_type, CIDR_rule, auth_token, product_key)  # Launches the EC2 instance
    except Exception as e:
      __LOGGER__.error("Unable to launch EC2 instance: '%s'. Please check AWS Console to make sure any EC2 instances launched have been terminated." % e)
      return

    try:
        # Once the EC2 instance starts up, the client still need to wait for graphlab to start up.
        # So allow for a larger than normal number of ping failures.
        __server__.start(num_tolerable_ping_failures=30)
    except Exception as e:
        __LOGGER__.error(e)
        __server__.stop()
        __server__ = None
        return

    num_tolerable_ping_failures = 3
    client = Client([], __server__.get_server_addr(), num_tolerable_ping_failures)
    auth_token = __server__.get_auth_token()
    if auth_token:
        client.add_auth_method_token(auth_token)
    client.start()
    glconnect._assign_server_and_client(__server__, client)


def set_credentials(access_key_id, secret_access_key):
    """
    Set the AWS credential environment variables:
    AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY.

    Parameters
    ----------
    access_key_id : string
        Value of AWS_ACCESS_KEY_ID

    secret_access_key : string
        Value of AWS_SECRET_ACCESS_KEY
    """
    os.environ['AWS_ACCESS_KEY_ID'] = str(access_key_id)
    os.environ['AWS_SECRET_ACCESS_KEY'] = str(secret_access_key)


@__catch_and_log__
def status():
    """
    Returns the status of a running, if any is up.
    """
    if __server__ is None:
        return 'No instance running.'
    aws_connection = boto.ec2.connect_to_region(__server__.region)
    status = aws_connection.get_all_instance_status(instance_ids=[__server__.instance_id])
    __LOGGER__.debug("EC2 status returned: %s" % status)
    if(not(len(status) == 1)):
        __LOGGER__.error('Invalid response from EC2. Unable to determine EC2 status. Please consult '
                         'the AWS EC2 Console.')

    return status[0].state_name


def terminate_EC2():
    """
    Brings down the current EC2 instance.
    """
    global __server__

    if __server__ is not None:
        __LOGGER__.debug("Stopping EC2 instance.")
        glconnect.stop()
        __server__ = None
    else:
        __LOGGER__.info("No EC2 instance to stop.")
