
from graphlab.data_structures.graph import Graph
import graphlab.toolkits.main as main
from graphlab.toolkits.model import Model


class KcoreModel(Model):
    """
    A Model object containing a coreid for each vertex, and the total number of
    cores in the graph.
    """
    def __init__(self, input_params, model):
        self.__proxy__ = model
        self._input = input_params
        self._fields = ['coreid', 'runtime', 'max_core', 'graph']

    def list_fields(self):
        """Return list of fields in the model."""

        return self._fields

    def get(self, field):
        """Return the value for the queried field."""

        if (field == 'coreid'):
            return Graph(_proxy=self.__proxy__.get('__graph__')).select_fields(['coreid']).get_vertices()
        elif (field == 'graph'):
            return Graph(_proxy=self.__proxy__.get('__graph__'))
        else:
            return self.__proxy__.get(field)

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""

        ret = {'hyperparams': self._input}
        for key in ['runtime', 'max_core']:
            ret[key] = self.get(key)
        return ret

    def _get_wrapper(self):
        return lambda m: KcoreModel(self._input, m)


def create(graph, kmin=0, kmax=100, verbose=False, plot=False):
    """
    Compute the K-core decomposition of the graph. Return a model object with
    total number of cores as well as the core id for each vertex in the graph.

    Parameters
    ----------
    graph : Graph
        The graph on which to compute the k-core decomposition.

    kmin : int, optional
        Minimun core id. Vertices having smaller core id than `kmin` will be
        assigned with coreid = `kmin`.

    kmax : int, optional
        Maximun core id. Vertices having larger core id than `kmax` will be
        assigned with coreid=`kmax`.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : Model
        A Model object that contains the coreid for each vertex.

    References
    ----------
    - Alvarez-Hamelin, J.I., et al. (2005) `K-Core Decomposition: A Tool for the
      Visualization of Large Networks <http://arxiv.org/abs/cs/0504107>`_.
    
    Examples
    --------
    If given an :class:`~graphlab.Graph` ``g``, we can create
    a :class:`~graphlab.kcore.KcoreModel` as follows:

    >>> kc = kcore.create(g)

    We can obtain the ``core id`` corresponding to each vertex in the graph
    ``g`` using: 

    >>> kcore_id = kc.get('coreid')     # SFrame
    """
    if not isinstance(graph, Graph):
        raise TypeError('graph input must be a Graph object.')

    if verbose is True:
        print "Starting K-core computation."
        verbose = 0

    opts = {'graph': graph.__proxy__, 'kmin': kmin, 'kmax': kmax}
    params = main.run('kcore', opts, verbose, plot)

    return KcoreModel({'kmin': kmin, 'kmax': kmax}, params['model'])
