"""
.. warning:: This product is currently in a beta release. The API reference is subject to change.

This package defines the GraphLab Graph, Vertex, and Edge objects. The Graph is
a directed graph, consisting of a set of Vertex objects and Edges that connect
pairs of Vertices. The methods in this module are available from the top level
import of the graphlab package.

"""

import graphlab.connect.main as glconnect
from graphlab.data_structures.sframe import SFrame
from graphlab.cython.graph import UnityGraphProxy
from graphlab.util import make_internal_url, shallow_throw
import pandas as pd

## \internal Default column name for vertex id.
_VID_COLUMN = '__id'

## \internal Default column name for source vid.
_SRC_VID_COLUMN = '__src_id'

## \internal Default column name for target vid.
_DST_VID_COLUMN = '__dst_id'

## \internal Flag indicates the existance of matplotlib and networkX.
_HAS_MATPLOTLIB = False
try:
    import networkx as nx
    import matplotlib.pyplot as plt
    _HAS_MATPLOTLIB = True
except:
    pass



#/**************************************************************************/
#/*                                                                        */
#/*                         Graph Related Classes                          */
#/*                                                                        */
#/**************************************************************************/
class Vertex(object):
    """
    A vertex object, consisting of a vertex ID and a dictionary of vertex
    attributes. The vertex ID can be an integer, string, or float.

    Parameters
    ----------
    vid : int | string | float
        Vertex ID.

    attr : dict
        Vertex attributes. A Dictionary of string keys and values with one of the
        following types: int, float, string, array of floats.

    _series : pandas.Series
        Internal. Do not use.

    Returns
    -------
    out : Vertex
        A new vertex object.
    """

    __slots__ = ['vid', 'attr']

    def __init__(self, vid, attr={}, _series=None):
        """__init__(self, vid, attr={})
        Construct a new vertex.
        """
        if not _series is None:
            self.vid = _series[_VID_COLUMN]
            self.attr = _series.to_dict()
            self.attr.pop(_VID_COLUMN)
        else:
            self.vid = vid
            self.attr = attr

    def __repr__(self):
        return "V(" + str(self.vid) + ", " + str(self.attr) + ")"

    def __str__(self):
        return "V(" + str(self.vid) + ", " + str(self.attr) + ")"


class Edge(object):
    """
    A directed edge between two vertices, consisting of source vertex ID,
    destination vertex ID, and a dictionary of edge attributes.

    Parameters
    ----------
    src_vid : int | string | float
        Source vertex ID.

    dst_vid : int | string | float
        Target vertex ID.

    attr : dicts
        Edge attributes. A Dictionary of string keys and values with one of the
        following types: integer, float, string, array of floats.

    _series : pandas.Series
        Do not use.

    Returns
    -------
    out : Edge
        A new edge object.
    """

    __slots__ = ['src_vid', 'dst_vid', 'attr']

    def __init__(self, src_vid, dst_vid, attr={}, _series=None):
        """__init__(self, vid, attr={})
        Construct a new edge.
        """
        if not _series is None:
            self.src_vid = _series[_SRC_VID_COLUMN]
            self.dst_vid = _series[_DST_VID_COLUMN]
            self.attr = _series.to_dict()
            self.attr.pop(_SRC_VID_COLUMN)
            self.attr.pop(_DST_VID_COLUMN)
        else:
            self.src_vid = src_vid
            self.dst_vid =  dst_vid
            self.attr = attr

    def __repr__(self):
        return ("E(" + str(self.src_vid) + " -> " + str(self.dst_vid) + ", " +
                str(self.attr) + ")")

    def __str__(self):
        return ("E(" + str(self.src_vid) + " -> " + str(self.dst_vid) + ", " +
                str(self.attr) + ")")


class Graph(object):
    """
    A directed graph that allows arbitrary dictionary attributes on vertices and
    edges. The constructor creates an empty graph, to which vertices and edges
    can be added. Alternatively, the :py:func:`load_graph` function creates a
    :class:`~graphlab.Graph` from a binary or text file.

    There are several ways to create a
    :class:`~graphlab.Graph`. The simplest is to make an empty graph then add
    vertices and edges. Note that Graphs are immutable, so a new graph is
    returned when vertices and edges are added.

    >>> import graphlab
    >>> from graphlab import Graph, Vertex, Edge, SFrame

    >>> g = Graph()

    >>> verts = [Vertex(0, attr={'breed': 'labrador'}),
                 Vertex(1, attr={'breed': 'labrador'}),
                 Vertex(2, attr={'breed': 'vizsla'})]
    >>> g = g.add_vertices(verts)
    >>> g = g.add_edges(Edge(1, 2))

    Because :class:`~graphlab.Graph` is immutable we can chain these steps
    together to make a new graph in a single line.

    >>> g = Graph().add_vertices([Vertex(i) for i in range(10)]).add_edges(
            [Edge(i, i+1) for i in range(9)])

    Graphs can also be created from an edge list stored in an
    :class:`~graphlab.SFrame`. Vertices are added to the graph automatically
    based on the edge list, and columns of the SFrame not used as source or
    destination vertex IDs are assumed to be edge attributes. For this example
    we download a dataset of James Bond characters to an SFrame, then build the
    graph.

    >>> data = SFrame.read_csv('http://s3.amazonaws.com/GraphLab-Datasets/bond/bond_edges.csv')
    >>> g = Graph()
    >>> g = g.add_edges(data, src_field='src', dst_field='dst')

    Finally, a :class:`~graphlab.Graph` can be created directly from a file,
    either local or remote, using the :py:func:`load_graph` method. Loading a
    graph with this method works with both the native binary save format and a
    variety of text formats.

    >>> g.save('james_bond')  # save file with the .graph extension
    >>> newgraph = graphlab.load_graph('james_bond.graph')

    Small graphs can be explored very efficiently with the :py:func:`show`
    method, which displays a plot of the graph. The vertex labels can be IDs or
    any vertex attribute.

    >>> g.show(vlabel='id', v_offset=0.05)

    For large graphs visual displays are difficult, but graph exploration can
    still be done with the :py:func:`summary` method---which prints the number
    of vertices and edges---or by retrieving subsets of edges and vertices.

    >>> print g.summary()
    {'num_edges': 21, 'num_vertices': 10}

    To retrieve the contents of a graph, the :py:func:`get_vertices` and
    :py:func:`get_edges` methods return SFrames. These functions can filter
    edges and vertices based on vertex IDs or attributes. Omitting IDs and
    attributes returns all vertices or edges.

    >>> verts = g.get_vertices(ids=['James Bond'])
            __id
    0  James Bond

    >>> bosses = g.get_edges(fields={'relation': 'worksfor'})
       relation       __src_id     __dst_id
    0  worksfor              M   James Bond
    1  worksfor              M   Moneypenny
    2  worksfor              M            Q
    3  worksfor  Elliot Carver  Henry Gupta
    4  worksfor  Elliot Carver    Gotz Otto

    >>> edges = g.get_edges()
    >>> print edges.head(5)
         relation        __src_id       __dst_id
    0    worksfor               M              Q
    1      friend  Inga Bergstorm     James Bond
    2   killed_by   Elliot Carver     James Bond
    3   engagedto    Paris Carver  Elliot Carver
    4  managed_by      Moneypenny              M


    See Also
    --------
    Vertex, Edge, SFrame
    """

    __slots__ = ['__proxy__']

    def __init__(self, _proxy=None):
        """__init__()
        Construct a new empty graph.
        """
        if (_proxy is None):
            self.__proxy__ = UnityGraphProxy(glconnect.get_client())
        else:
            self.__proxy__ = _proxy

    def __str__(self):
        """Returns a readable string representation summarizing the graph."""
        return "Graph(%s)" % str(self.summary())

    def __repr__(self):
        """Returns a readable string representation summarizing the graph."""
        return "Graph(%s)\nFields:%s" % (str(self.summary()), str(self.get_fields()))

    def summary(self):
        """Return basic graph statistics in a dictionary.

        Returns
        -------
        out : dict
            A dictionary containing the number of vertices and edges.
        """
        ret = self.__proxy__.summary()
        return dict(ret.items())

    @shallow_throw
    def get_vertices(self, ids=[], fields={}, format='sframe'):
        """
        Return a collection of vertices and their attributes.

        Parameters
        ----------

        ids : List [int | float | str]
            List of vertex IDs to retrieve. Only vertices in this list will be
            returned.

        fields : dict | pandas.DataFrame
            Dictionary specifying equality constraint on field values. For
            example ``{'gender': 'M'}``, returns only vertices whose 'gender'
            field is 'M'. ``None`` can be used to designate a wildcard. e.g.
            {'relationship': None} will find all vertices with the field
            'relationship' regardless of the value.

        format : {'sframe', 'list'}
            Output format. The SFrame output (default) contains a column
            __src_id with vertex IDs and a column for each vertex attribute.
            List output returns a list of Vertex objects.

        Returns
        -------
        out : Vertex list, pandas.DataFrame
            A Vertex list or a pandas dataframe of vertices.

        Examples
        --------
        Return all vertices in the graph.

        >>> g.get_vertices()

        Return vertex with ID of 1.

        >>> g.get_vertices(ids=1)

        Return vertices 1, 2, and 3.

        >>> g.get_vertices(ids=[1, 2, 3])

        Return vertices with the vertex attribute "gender".

        >>> g.get_vertices(fields={'gender': None})

        Return vertices with the vertex attribute "gender" equal to "M".

        >>> g.get_vertices(fields={'gender': 'M'})

        Return vertices 1, 2, 3 which have the vertex attribute "gender" set to
        "M".

        >>> g.get_vertices(ids=[1, 2, 3], fields={'gender': 'M'})
        """
        if not hasattr(ids, '__iter__'):
            ids = [ids]

        sf = SFrame(_proxy=self.__proxy__.get_vertices(ids, fields))
        if (format == 'sframe'):
            return sf
        elif (format == 'dataframe'):
            if sf.num_rows() == 0:
                return pd.DataFrame()
            else:
                df = sf.head(sf.num_rows())
                return df.set_index('__id')
        elif (format == 'list'):
            return _dataframe_to_vertex_list(sf.head(sf.num_rows()))
        else:
            raise ValueError("Invalid format specifier")

    @shallow_throw
    def get_edges(self, src_ids=[], dst_ids=[], fields={}, format='sframe'):
        """
        Return a collection of edges and their attributes. This function can be
        used to find edges by vertex IDs, filter on edge attributes, or list
        in-out neighbors of graphs.

        Parameters
        ----------
        src_ids, dst_ids : list
            Parallel arrays of vertex IDs, with each pair corresponding to an
            edge to fetch. Only edges in this list are returned. ``None`` can be
            used to designate a wild-card. For instance, ``src_ids=[1, 2,
            None]``, ``dst_ids=[3, None, 5]`` will fetch the edge 1->3, all
            outgoing edges of 2 and all incoming edges of 5. src_id and dst_id
            may be left empty, which implies an array of all wildcards.

        fields : dict
            Dictionary specifying equality constraints on field values. For
            example ``{'relationship': 'following'}``, returns only edges whose
            'relationship' field equals 'following'. ``None`` can be used as a
            value to designate a wildcard. e.g. ``{'relationship': None}`` will
            find all edges with the field 'relationship' regardless of the
            value.

        format : {'sframe', 'list'}
            Output format. The 'sframe' output (default) contains columns
            __src_id and __dst_id with edge vertex IDs and a column for each
            edge attribute. List ouput returns a list of Edge objects.

        Returns
        -------
        out : SFrame | list [Edge]
            An SFrame or list of edges.

        Examples
        --------
        Return all edges in the graph.

        >>> g.get_edges()
        >>> g.get_edges(src_ids=[None], dst_ids=[None])

        Return edges with attribute "rating".

        >>> g.get_edges(fields={'rating': None})

        Return edges with the attribute "rating" of 5.

        >>> g.get_edges(fields={'rating': 5})

        Return edges 1 --> 3 and 2 --> 4 (if present in the graph).

        >>> g.get_edges(src_ids=[1, 2], dst_ids=[3, 4])

        Return edges 1 --> 3 and 2 --> 4 which have the edge attribute "rating"
        of 5.

        >>> g.get_edges(src_ids=[1, 2], dst_ids=[3, 4], fields={'rating': 5})

        Return out-edges of vertex 1 with the edge attribute "rating".

        >>> g.get_edges(src_ids=[1], dst_ids=[None], fields={'rating': None})

        Return in-edges of vertex 3.

        >>> g.get_edges(src_ids=[None], dst_ids=[3])

        Return out-edges of 1 and in-edges of 3. If edge 1 --> 3 exists, it will
        only be returned once.

        >>> g.get_edges(src_ids=[1, None], dst_ids=[None, 3])
        """
        if not hasattr(src_ids, '__iter__'):
            src_ids = [src_ids]
        if not hasattr(dst_ids, '__iter__'):
            dst_ids = [dst_ids]

        # implicit Nones
        if len(src_ids) == 0 and len(dst_ids) > 0:
            src_ids = [None] * len(dst_ids)
        # implicit Nones
        if len(dst_ids) == 0 and len(src_ids) > 0:
            dst_ids = [None] * len(src_ids)

        sf = SFrame(_proxy=self.__proxy__.get_edges(src_ids, dst_ids, fields))
        if (format == 'sframe'):
            return sf
        if (format == 'dataframe'):
            if sf.num_rows() == 0:
                return pd.DataFrame()
            else:
                return sf.head(sf.num_rows())
        elif (format == 'list'):
            return _dataframe_to_edge_list(sf.head(sf.num_rows()))
        else:
            raise ValueError("Invalid format specifier")

    @shallow_throw
    def add_vertices(self, vertices, vid_field=None, format='auto'):
        """
        Add one or many vertices to the graph. Vertices should be input as a
        list of Vertex objects (if only one Vertex is added this will be a list
        of length 1), a pandas DataFrame, or an SFrame. If vertices are
        specified by SFrame or DataFrame, vid_field specifies which column
        contains the vertex ID. Remaining columns are assumed to hold additional
        vertex attributes. If these attributes are not already present in the
        graph, they are added, possibly overwriting existing vertices.

        Parameters
        ----------
        vertices : Vertex | List [Vertex] | pandas.Dataframe | SFrame
            Vertex data.

        format : {'auto', 'dataframe', 'sframe', 'list', 'vertex'}
            Format of the vertices parameter. If set to 'auto' (default), the
            method automatically detects the format.

        vid_field : string
            Field in the Dataframe or SFrame to use as vertex ID. Required if
            vertices is an SFrame. If vertices is a DataFrame and vid_field is
            not specified, the row index is used as vertex ID.

        Returns
        -------
        out : Graph
            A new graph with vertices added.

        Raises
        ------
        ValueError
            If the arguments do not match the required type.
        """
        if format == 'auto':
            if isinstance(vertices, pd.DataFrame):
                format = 'dataframe'
            elif isinstance(vertices, SFrame):
                format = 'sframe'
            elif isinstance(vertices, list):
                format = 'list'
            elif isinstance(vertices, Vertex):
                format = 'vertex'
            else:
                raise TypeError('Vertices type %s is Not supported.' % str(type(vertices)))

        if format == 'vertex':
            return Graph(_proxy=self.__proxy__.add_vertices(
                  _vertex_list_to_dataframe([vertices], '__id'), '__id'))
        elif format == 'list':
            return Graph(_proxy=self.__proxy__.add_vertices(
                         _vertex_list_to_dataframe(vertices, '__id'), '__id'))
        elif format == 'dataframe':
            if vid_field is None:
                # using the dataframe index as vertex id
                if vertices.index.is_unique:
                    if not ("index" in vertices.columns):
                        # pandas reset_index() will insert a new column of name "index".
                        return Graph(_proxy=self.__proxy__.add_vertices(vertices.reset_index(), "index"))
                    else:
                        # pandas reset_index() will insert a new column of name "level_0" if there exists a column named "index".
                        return Graph(_proxy=self.__proxy__.add_vertices(vertices.reset_index(), "level_0"))
                else:
                    raise ValueError("Index of the vertices dataframe is not unique, \
                            try specifying vid_field name to use a column for vertex ids.")
            else:
                # using the given column as vertex id
                return Graph(_proxy=self.__proxy__.add_vertices(vertices, vid_field))
        elif format == 'sframe':
            if vid_field is not None:
                return Graph(_proxy=self.__proxy__.add_vertices_from_sframe(vertices.__proxy__, vid_field))
                pass
            else:
                raise ValueError("vid_field must be specified for sframe input.")
        else:
            raise TypeError('Vertices type %s is Not supported.' % str(type(vertices)))

    @shallow_throw
    def add_edges(self, edges, src_field=None, dst_field=None, format='auto'):
        """
        Add one or many edges to the graph. Edges should be input as a list of
        Edge objects (for a single edge, this will be a list of length 1), an
        SFrame, or a Pandas DataFrame. If the new edges are in an SFrame or
        DataFrame, then `src_field` and `dst_field` are required to specify the
        columns that contain the source and target vertex IDs; additional
        columns are treated as edge attributes. If the new edges contain
        attributes not already present in the graph, new graph attributes are
        created in the graph. This may overwrite existing edges that lack the
        new attributes.

        Parameters
        ----------
        edges : Edge | List [Edge] | pandas.Dataframe | SFrame
            Edge data. If the edges are in an SFrame or DataFrame, then
            `src_field` and `dst_field` are required to specify the columns that
            contain the source and target vertex IDs.

        format : {'auto', 'dataframe', 'list', 'sframe', 'edge'}
            Format of the edges parameter. If set to 'auto' (default), the
            method automatically detects the type of the edges argument.

        src_field : string
            Field in the Pandas DataFrame or SFrame to use as source vertex IDs.
            Not required if edges is a list.

        dst_field : string
            Field in the Pandas DataFrame or SFrame to use as target vertex IDs.
            Not required if edges is a list.

        Returns
        -------
        out : Graph
            A new graph with edges added.

        Raises
        ------
        ValueError
            If the arguments do not match the required types.
        """

        if format == 'auto':
            if isinstance(edges, pd.DataFrame):
                format = 'dataframe'
            elif isinstance(edges, SFrame):
                format = 'sframe'
            elif isinstance(edges, list):
                format = 'list'
            elif isinstance(edges, Edge):
                format = 'edge'
            else:
                raise TypeError('Vertices type %s is Not supported.' % str(type(edges)))

        if format == 'edge':
            return Graph(_proxy=self.__proxy__.add_edges(
                _edge_list_to_dataframe([edges], '__src_vid', '__dst_vid'), '__src_vid', '__dst_vid'))
        elif format == 'list':
            return Graph(_proxy=self.__proxy__.add_edges(
                _edge_list_to_dataframe(edges, '__src_vid', '__dst_vid'), '__src_vid', '__dst_vid'))
        elif format == 'dataframe':
            if src_field is None:
                raise ValueError("src_field must be specified for Pandas input")
            if dst_field is None:
                raise ValueError("dst_field must be specified for Pandas input")
            return Graph(_proxy=self.__proxy__.add_edges(edges, src_field, dst_field))
        elif format == 'sframe':
            if src_field is None:
                raise ValueError("src_field must be specified for SFrame input")
            if dst_field is None:
                raise ValueError("dst_field must be specified for SFrame input")
            return Graph(_proxy=self.__proxy__.add_edges_from_sframe(edges.__proxy__, src_field, dst_field))
        else:
            raise TypeError('Edges type %s is Not supported.' % str(type(edges)))

    def get_fields(self):
        """Return a list of vertex and edge attribute fields in the graph."""
        return self.__proxy__.get_fields()

    @shallow_throw
    def select_fields(self, fields):
        """
        Return a new graph with only the selected fields.

        Notes
        -----
            Non-existent fields are ignored.

        Parameters
        ----------
        fields : list [string]
            A list of field names to select.

        Returns
        -------
        out : Graph
            A new graph whose vertex and edge data are projected to the selected
            fields.

        """
        return Graph(_proxy=self.__proxy__.select_fields(fields))

    @shallow_throw
    def delete_field(self, field):
        """
        Delete a field in the graph. The field is removed from all vertices and
        edges.

        Parameters
        ----------
        field : string
            Name of the field to be deleted.

        Returns
        -------
        out : Graph
            The new graph with field removed.

        Examples
        --------
        >>> g = Graph().add_vertices([Vertex('cat', {'fluffy': 1}),
                                     Vertex('dog', {'fluffy': 1, 'woof': 1}),
                                     Vertex('hippo', {})])
        >>> g2 = g.delete_field('woof')
        >>> g2.get_vertices()
        """
        return Graph(_proxy=self.__proxy__.delete_field(field))

    @shallow_throw
    def copy_field(self, src_field, new_field):
        """
        Create a new field by copying an existing field. All vertices and edges
        which contain the attribute src_field will now have a new attribute with
        name new_field and value equal to the src_field attribute.

        Parameters
        ----------
        src_field : string
            Name of the source field.

        new_field : string
            Name of the destination field.

        Returns
        -------
        out : Graph
            The new graph with copied field.

        Examples
        --------
        >>> g = Graph().add_vertices([Vertex('cat', {'fluffy': 1}),
                                     Vertex('dog', {'fluffy': 1, 'woof': 1}),
                                     Vertex('hippo', {})])
        >>> g2 = g.copy_field('woof', 'bark')
        >>> g2.get_vertices()
        """
        return Graph(_proxy=self.__proxy__.copy_field(src_field, new_field))

    @shallow_throw
    def save(self, filename, format="binary"):
        """
        Save the graph to disk. If the graph is saved in binary format, the
        graph can be loaded using the ``load_graph`` method. Alternatively, it
        can be saved as "json" to get a human readable / portable graph
        representation.

        Parameters
        ----------
        filename : string
            Filename to use when saving the file. The file extensions
            ".graph" and ".json" are appended automatically if not already in
            filename.

        format : {'binary', 'json'}
            File format.

        Raises
        ------
        ValueError
            If the format is not supported.

        See Also
        --------
        load_graph
        """

        if format not in ['binary', 'json']:
            raise ValueError('Invalid format: %s. Supported formats are: %s'
                             % (format, ['binary', 'json']))

        # automatically correct the format based on the filename suffix
        if filename.endswith('.graph'):
            format = 'binary'
        elif filename.endswith(('.json', '.json.gz')):
            format = 'json'
        elif format is 'binary':
            filename = filename + ".graph"
        elif format is 'json':
            filename = filename + '.json'
        self.__save_graph(make_internal_url(filename), format)

    def __save_graph(self, filename, format):
        if format == "json":
            assert filename.endswith('.json')
            self.__proxy__.save_graph_as_json(filename)
        elif format == "binary":
            assert filename.endswith('.graph')
            self.__proxy__.save_graph(filename)
        else:
            raise ValueError("Unsupported format: " + format)

    def show(self, vlabel=None, vcolor=[0.6, 0., 0.6], highlight=[],
             highlight_color=[1., 0.5, 0.], node_size=300, elabel=None,
             ecolor='gray', ewidth=3, v_offset=0.03, h_offset=0.):
        """
        Draw the graph using the `NetworkX <http://networkx.github.io/>`_ spring
        layout. Beacause drawing big graphs is messy and resource consuming,
        this function only works for graphs with fewer than 1,000 edges.

        Parameters
        ----------
        vlabel : string
            Field name for the label on each vertex. The default is None, which
            omits vertex labels. Set to 'id' to use the vertex ID as the label.

        vcolor : list
            RGB triple for the primary vertex color. Default is orange.

        highlight : list
            Vertex IDs to highlight in a different color.

        highlight_color : list
            RGB triple for the color of highlighted vertices. Default is purple.

        node_size : int
            Size of plotted vertices.

        elabel : string
            Field name for the label on each edge.

        ecolor : string
            Edge color.

        ewidth : int
            Edge width.

        v_offset : float
            Vertical offset of vertex labels, as a fraction of total plot
            height. For example, the default of 0.03 moves the label 3% of the
            plot height higher in the canvas.

        h_offset : float
            Horizontal offset of vertex labels, as a fraction of total plot
            width. For example, an offset of 0.03 moves the label 3% of the plot
            width to the right. Default is 0.0.
        """

        ## Error trapping
        if not _HAS_MATPLOTLIB:
            raise RuntimeError("Was unable to import matplotlib or networkx. Unable to show graph.")

        maxedges = 1000
        num_edges = self.summary()["num_edges"]
        if (num_edges > maxedges):
            raise RuntimeError("Cannot show graph with too many edges")

        ## Make the NetworkX graph
        nxgraph = nx.Graph()
        vertices = self.get_vertices(format='list')
        edges = self.get_edges(format='list')
        node_color = []
        nodelist = []

        for v in vertices:
            nodelist.append(v.vid)
            nxgraph.add_node(v.vid)

            if v.vid in highlight:
                node_color.append(vcolor)
            else:
                node_color.append(highlight_color)

        for e in edges:
            nxgraph.add_edge(e.src_vid, e.dst_vid)

        pos = nx.spring_layout(nxgraph)

        ## Draw the graph
        nx.draw_networkx_nodes(nxgraph, pos, nodelist=nodelist,
                               node_color=node_color, node_size=node_size,
                               alpha=0.7)

        nx.draw_networkx_edges(nxgraph, pos, edge_color=ecolor, width=ewidth,
                               alpha=0.5)

        ## Draw vertex labels
        if vlabel:
            # get offset positions
            pos_label = {}
            for k, v in pos.items():
                xcoord = max(0., pos[k][0] + h_offset)
                ycoord = max(0., pos[k][1] + v_offset)
                pos_label[k] = [xcoord, ycoord]

            # plot labels
            if vlabel == "id":
                names = {v.vid: v.vid for v in vertices}
                nx.draw_networkx_labels(nxgraph, pos_label, labels=names,
                                        font_color='b')
            else:
                names = {v.vid: v.attr[vlabel] for v in vertices}
                nx.draw_networkx_labels(nxgraph, pos_label, labels=names,
                                        font_color='b')

        # Draw edge labels
        if elabel:
            names = dict([((e.src_vid, e.dst_vid), e.attr[elabel]) for e in edges])
            nx.draw_networkx_edge_labels(nxgraph, pos, edge_labels=names,
                                         font_color='b')

        plt.axis('off')
        plt.show()


@shallow_throw
def load_graph(filename, format='binary'):
    """
    Load graph from text file or previously saved graph binary.

    Parameters
    ----------
    filename : string
        Location of the file. Can be a local path or a remote URL.

    format : {'binary', 'snap', 'csv', 'tsv', 'adj'}
        Data format.

        - 'binary' is the native graph format obtained from `Graph.save`.
        - 'snap' is a tab separated edge list format with comments, used in
          the `Stanford Network Analysis Platform <http://snap.stanford.edu/snap/>`_.
        - 'csv' is a comma-separated edge list without header or comments.
        - 'tsv' is a tab-separated edge list without header or comments.
        - 'adj' is an adjacency list in the format of [source vertex ID] [number
          of target vertices] [target ID 1] [target ID 2] ...
    """
    if not format in ['binary', 'snap', 'csv', 'tsv', 'adj']:
        raise ValueError('Invalid format: %s' % format)

    if format is 'binary' and not filename.endswith('.graph'):
        filename = filename + '.graph'

    try:
        if format is 'binary':
            proxy = glconnect.get_unity().load_graph(make_internal_url(filename))
        else:
            proxy = glconnect.get_unity().load_graph_from_url(make_internal_url(filename), format)

        return Graph(_proxy=proxy)
    except IOError as e:
        raise e


def _vertex_list_to_dataframe(ls, id_column_name):
    """
    Convert a list of vertices into dataframe.
    """
    cols = reduce(set.union, (set(v.attr.keys()) for v in ls))
    df = pd.DataFrame({id_column_name: [v.vid for v in ls]})
    for c in cols:
        df[c] = [v.attr.get(c) for v in ls]
    return df


def _edge_list_to_dataframe(ls, src_column_name, dst_column_name):
    """
    Convert a list of edges into dataframe.
    """
    cols = reduce(set.union, (set(e.attr.keys()) for e in ls))
    df = pd.DataFrame({
        src_column_name: [e.src_vid for e in ls],
        dst_column_name: [e.dst_vid for e in ls]})
    for c in cols:
        df[c] = [e.attr.get(c) for e in ls]
    return df


def _dataframe_to_vertex_list(df):
    """
    Convert dataframe into list of vertices, assuming that vertex ids are stored in _VID_COLUMN.
    """
    cols = df.columns
    if len(cols):
        assert _VID_COLUMN in cols, "Vertex DataFrame must contain column %s" % _VID_COLUMN
        df = df[cols].T
        ret = [Vertex(None, _series=df[col]) for col in df]
        return ret
    else:
        return []


def _dataframe_to_edge_list(df):
    """
    Convert dataframe into list of edges, assuming that source and target ids are stored in _SRC_VID_COLUMN, and _DST_VID_COLUMN respectively.
    """
    cols = df.columns
    if len(cols):
        assert _SRC_VID_COLUMN in cols, "Vertex DataFrame must contain column %s" % _SRC_VID_COLUMN
        assert _DST_VID_COLUMN in cols, "Vertex DataFrame must contain column %s" % _DST_VID_COLUMN
        df = df[cols].T
        ret = [Edge(None, None, _series=df[col]) for col in df]
        return ret
    else:
        return []
