
from graphlab.data_structures.graph import Graph
import graphlab.toolkits.main as main
from graphlab.toolkits.model import Model


class PagerankModel(Model):
    """
    A Model object containing the pagerank value for each vertex.
    """
    def __init__(self, input_params, model):
        self.__proxy__ = model
        self._input = input_params
        self._fields = ['pagerank', 'runtime', 'total_pagerank', 'graph']

    def list_fields(self):
        """Return the list of fields in the model."""

        return self._fields

    def get(self, field):
        """Return the value for the queried field."""

        if (field == 'pagerank'):
            return Graph(_proxy=self.__proxy__.get('__graph__')).select_fields(['pagerank']).get_vertices()
        elif(field == 'graph'):
            return Graph(_proxy=self.__proxy__.get('__graph__'))
        else:
            return self.__proxy__.get(field)

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""

        ret = {'hyperparams': self._input}
        for key in ['runtime', 'total_pagerank', 'delta']:
            ret[key] = self.get(key)
        return ret

    def _get_wrapper(self):
        return lambda m: PagerankModel(self._input, m)


def create(graph, reset_prob=0.15, threshold=0.01, verbose=False, plot=False):
    """
    Compute the PageRank for each vertex in the graph. Return a model object
    with total PageRank as well as the PageRank value for each vertex in the
    graph.

    Parameters
    ----------
    graph : Graph
        The graph on which to compute the pagerank value.

    reset_prob : float, optional
        Probability that a random surfer jumps to an arbitrary page.

    threshold : float, optional
        Threshold for convergence in absolute value.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : Model
        A Model object that contains the total and individual PageRank.

    References
    ----------
    - `Wikipedia - PageRank <http://en.wikipedia.org/wiki/PageRank>`_
    - Page, L., et al. (1998) `The PageRank Citation Ranking: Bringing Order to
      the Web <http://ilpubs.stanford.edu:8090/422/1/1999-66.pdf>`_.
    
    Examples
    --------
    If given an :class:`~graphlab.Graph` ``g``, we can create
    a :class:`~graphlab.pagerank.PageRankModel` as follows:

    >>> pr = pagerank.create(g)
 
    We can obtain the page rank corresponding to each vertex in the graph ``g``
    using: 

    >>> pr_out = pr.get('pagerank')     # SFrame
    """
    if not isinstance(graph, Graph):
        raise TypeError('graph input must be a Graph object.')

    if plot is True:
        print "The plot functionality for pagerank is not yet implemented."
        plot = False

    if verbose is True:
        print "Starting pagerank computation."
        verbose = 0

    opts = {'threshold': threshold, 'reset_prob': reset_prob, 'graph': graph.__proxy__}
    params = main.run('pagerank', opts, verbose, plot)
    return PagerankModel({'threshold': threshold, 'reset_prob': reset_prob},
                         params['model'])
