"""@package graphlab.toolkits

Defines a basic interface for a model object.
"""

from graphlab.util import make_internal_url
import graphlab.connect.main as glconnect


def load_model(filename):
    """
    Load a model. Assumes the model is in binary format, as saved by
    Model.save().

    Parameters
    ----------
    filename : string
        Location of the file to load. Can be a local path or a remote URL.
    """
    return glconnect.get_unity().load_model(make_internal_url(filename))


class Model(object):
    """
    This class defines the minimal interface of a model object
    storing the results from a toolkit function.

    User can query the model object via the `get` function, and
    the list of the queriable fields can be obtained via `list_fields`.
    """

    """The UnityModel Proxy Object"""
    __proxy__ = None

    def name(self):
        """Return the name of the model"""
        return self.__class__.__name__

    def list_fields(self):
        """Return list of fields in the model."""
        raise NotImplementedError

    def get(self, field):
        """Return the value for the queried field."""
        raise NotImplementedError

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""
        raise NotImplementedError

    def _get_wrapper(self):
        """Return a lambda function: UnityModel -> M, for constructing model class M from a UnityModel proxy."""
        raise NotImplementedError

    def save(self, filename):
        """
        Save the model to file in binary format.

        Parameters
        ----------
        filename : string
            Target destination for the model. Can be a local filename or a
            remote URL.
        """
        return glconnect.get_unity().save_model(self, make_internal_url(filename))

    def __repr__(self):
        ret = "Class: " + self.name() + "\n"
        ret += "Summary:\n"
        summary = self.summary()
        for k in summary:
            ret += "\t %s: %s\n" % (k, summary[k])
        ret += "Available fields for get():\n\t" + str(self.list_fields()) + "\n"
        ret += "\n"
        return ret
