"""
This package provides the GraphLab recommender system toolkit.

The toolkit includes a module with several recommender system models that can be
trained and used to make new recommendations. For detail beyond this overview,
please see the tutorials at http://www.graphlab.com.

The GraphLab recommender models accept an
:class:`~graphlab.SFrame` or a Pandas DataFrame. Our
example dataset is stored in a csv file, which is retrieved from an Amazon S3
address. The most basic type of dataset has just two columns: user ID and item
ID. The fastest way to use the GraphLab recommender system is the default matrix
factorization model, accessed with the :py:func:`create` function.

.. sourcecode:: python

    >>> from graphlab import recommender, SFrame
    >>> data = SFrame('s3:///graphlab-sample-data/example.csv')
    >>> m = recommender.create(data, user='user_id', item='item_id')
    >>> recs = m.recommend(k=5, user_ids=['u1', 'u3', 'u15'])
    >>> print recs

The default GraphLab recommender uses a matrix factorization model with fixed
values for the hyperparameters. The matrix factorization model can also be
called explicitly, with much greater flexibility. The
:py:func:`~graphlab.matrix_factorization.create` method both constructs a model
and trains it on data; it returns a
:py:class:`~graphlab.matrix_factorization.MatrixFactorizationModel`.

.. sourcecode:: python

    >>> from graphlab import matrix_factorization
    >>> m = matrix_factorization.create(data, user='user_id', item='item_id',
                                        D=7, regularizer=0.05, nmf=True,
                                        use_bias=False)

The GraphLab recommender toolkit includes other model classes as well. The
:py:class:`~graphlab.item_similarity.ItemSimilarityModel` computes the
similarity between each pair of items and recommends items to each user that
are closest to items she has already used or liked. It measures item similarity
with either Jaccard or cosine distance.

.. sourcecode:: python

    >>> from graphlab import item_similarity
    >>> m = item_similarity.create(data, user='user_id', item='item_id',
                                   similarity_type='jaccard')


All model objects in the :mod:`recommender` module expose a common set of
methods. With the :class:`~graphlab.item_similarity.ItemSimilarityModel`, for
example, the following methods are available:

.. currentmodule:: graphlab.item_similarity

.. autosummary::

    ItemSimilarityModel.evaluate
    ItemSimilarityModel.get
    ItemSimilarityModel.get_validation_split
    ItemSimilarityModel.list_fields
    ItemSimilarityModel.name
    ItemSimilarityModel.predict
    ItemSimilarityModel.recommend
    ItemSimilarityModel.save
    ItemSimilarityModel.summary
    ItemSimilarityModel.training_stats

See http://www.graphlab.com/products/create/tutorials.html for tutorials on 
building and using recommenders.

"""

from graphlab.toolkits.recommender import matrix_factorization
from graphlab.toolkits.recommender import popularity
from graphlab.toolkits.recommender import item_means
from graphlab.toolkits.recommender import item_similarity
from graphlab.toolkits.recommender import linear_regression
from graphlab.toolkits.recommender import logistic_regression

from graphlab.toolkits.recommender.util import create
from graphlab.toolkits.recommender.util import compare_models
from graphlab.toolkits.recommender.util import rmse
