"""
Methods for creating a model that predicts using the mean item rating.
"""

import graphlab as _graphlab
from graphlab.toolkits.recommender.recommender import RecommenderModel
from graphlab.data_structures.sframe import SFrame as _SFrame
from pandas import DataFrame as _DataFrame


def create(dataset, user, item, target,
           holdout_probability=0.0,
           verbose=True, plot=False, random_seed=0, **kwargs):
    """
    Create an :class:`~graphlab.item_means.ItemMeansModel` that uses each item's mean rating.
    These mean ratings are used when scoring (user, item) pairs and making recommendations, regardless of user.

    Parameters
    ----------
    dataset : pandas.DataFrame/SFrame
        The dataset to use for training the model.

    user : string
        The column name of the dataset that corresponds to user id.

    item : string
        The column name of the dataset that corresponds to item id.

    target : string
        The column name of the dataset that corresponds to the target of
        interest, e.g. the rating a user gave a particular item.

    holdout_probability : float, optional
        Proportion of the dataset held out of model training. The held-out
        subset is used to estimate the model's error rate when making
        predictions with new data; the prediction is available in the model's
        training_stats field.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    random_seed : integer, optional
        The random seed used to choose the training and validation
        split if holdout_probability is nonzero.

    Returns
    -------
    out : ItemMeansModel
        A trained ItemMeansModel.

    Examples
    --------
    If given an :class:`~graphlab.SFrame` ``sf`` with columns ``user_id`` and ``item_id``, then we can
    create a :class:`~graphlab.item_means.ItemMeansModel` as follows:

    >>> m = item_means.create(sf, 'user_id', 'item_id')

    With this model object one can make recommendations for the unique users in ``sf``:

    >>> recs = m.recommend(sf)

    The model can be saved to disk as follows:

    >>> m.save(filename)

    For more, see the documentation for :class:`~graphlab.item_means.ItemMeansModel`.

    """
    if not isinstance(dataset, (_DataFrame, _SFrame)):
        raise TypeError('dataset input must be a pandas.DataFrame or SFrame')

    if type(dataset) != _SFrame:
        dataset = _SFrame(dataset)

    if plot is True:
        print "The plot functionality for item means is not yet implemented."
        plot = False

    opts = {'model_name': 'item_counts'}
    response = _graphlab.toolkits.main.run("recsys_init", opts)
    model_proxy = response['model']

    opts = {'user_column': user,
            'item_column': item,
            'target_column': target,
            'training_data': dataset,
            'random_seed' : random_seed,
            'model': model_proxy,
            'holdout_probability': holdout_probability}
    opts.update(kwargs)
    response = _graphlab.toolkits.main.run("recsys_train", opts, verbose, plot)

    return ItemMeansModel(model_proxy)


class ItemMeansModel(RecommenderModel):
    """
    Use each item's mean rating to predict the target variable, regardless of the user.
    """

    def __init__(self, model_proxy):
        self.__proxy__ = model_proxy

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return ItemMeansModel(model_proxy)
        return model_wrapper
