"""
Methods for performing matrix factorization.  See
graphlab.matrix_factorization.create for additional documentation.
"""

import graphlab as _graphlab
from graphlab.toolkits.recommender.recommender import RecommenderModel
from graphlab.data_structures.sframe import SFrame as _SFrame
from pandas import DataFrame as _DataFrame


def create(dataset, user, item, target, D=10, reg=0.01, nmf=False,
           use_bias=True, holdout_probability=0.0, niter=10,
           verbose=True, plot=False, random_seed=0, **kwargs):
    """
    Create a :class:`~graphlab.matrix_factorization.MatrixFactorizationModel`.
    A trained model can be used to score (user, item) pairs and make recommendations.

    This is an implementation of basic Alternating Least Squares with
    user bias, item bias and a global bias term.

    References
    ----------
    See this paper for details about the model:
    Koren, Yehuda, Robert Bell and Chris Volinsky.
    "Matrix Factorization Techniques for Recommender Systems."
    Computer Volume: 42, Issue: 8 (2009): 30-37

    Parameters
    ----------
    dataset : pandas.DataFrame/SFrame
        The dataset to use for training the model.

    user : string
        The column name of the dataset that corresponds to user id.

    item : string
        The column name of the dataset that corresponds to item id.

    target : string
        The model will be trained to predict this column of the data set.

    D : int, optional
        Number of latent factors to use for describing each item (default is
        10).

    reg : float, optional
        Regularization weight (default is 0.01)

    nmf : boolean, optional
        Non-negative matrix factorization flag. If True, the model is constrained
        to use only non-negative matrices (default is False).

    use_bias : boolean, optional
        Bias term flag. If True (default), the model includes user and item bias
        terms.

    holdout_probability : float, optional
        Proportion of the dataset held out of model training. The held-out
        subset is used to estimate the model's error rate when making
        predictions with new data; the prediction is available in the model's
        training_stats field.

    niter : int, optional
        Number of optimization iterations to run.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    random_seed : integer, optional
        The random seed used to choose the training and validation
        split if holdout_probability is nonzero.

    Returns
    -------
    out : MatrixFactorizationModel
        A trained model.

    Examples
    --------
    If given an :class:`~graphlab.SFrame` ``sf`` with columns ``user_id`` and ``item_id`` and
    a column we would like to predict called ``target``, then we can
    create a :class:`graphlab.matrix_factorization.MatrixFactorizationModel` as follows:

    >>> m = matrix_factorization.create(sf, 'user_id', 'item_id', target='target')

    With this model object one can make recommendations for the unique users in ``sf``:

    >>> recs = m.recommend(sf)

    The model can be saved to disk as follows:

    >>> m.save(filename)

    For more, see the documentation for :class:`~graphlab.matrix_factorization.MatrixFactorizationModel`.


    """
    if not isinstance(dataset, (_DataFrame, _SFrame)):
        raise TypeError('dataset input must be a pandas.DataFrame or SFrame')

    if type(dataset) != _SFrame:
        dataset = _SFrame(dataset)

    opts = {'model_name': 'matrix_factorization',
            'D': D,
            'regularizer': reg,
            'nmf': nmf,
            'use_bias': use_bias}

    response = _graphlab.toolkits.main.run("recsys_init", opts)
    model_proxy = response['model']

    opts = {'user_column': user,
            'item_column': item,
            'target_column': target,
            'training_data': dataset,
            'model': model_proxy,
            'random_seed' : random_seed,
            'holdout_probability': holdout_probability,
            'niter': niter}
    opts.update(kwargs)
    response = _graphlab.toolkits.main.run("recsys_train", opts, verbose, plot)

    return MatrixFactorizationModel(model_proxy)


class MatrixFactorizationModel(RecommenderModel):
    """
    A matrix factorization model is able to learn a set of parameters for each user and for each item, scoring a (user, item) pair based on the similarity of the user and the item.
    In other words, the method will compose a set of user-item ratings (when represented as a matrix) into a
    number of latent factors for each item and corresponding weights for each
    user.
    """

    def __init__(self, model_proxy):
        self.__proxy__ = model_proxy

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return MatrixFactorizationModel(model_proxy)
        return model_wrapper
