from config import DEFAULT_CONFIG as CONFIG
from metric_mock import MetricMock

# sets default timeout to 3 seconds
import socket
socket.setdefaulttimeout(3)

# metrics libraries
import mixpanel
import librato

import Queue
import logging
import os
import platform
import pprint
import threading
import time
import uuid
__ALL__ = [ 'MetricTracker' ]

try:
  from psutil import TOTAL_PHYMEM, NUM_CPUS
except ImportError:
  TOTAL_PHYMEM = 0
  NUM_CPUS = 0

# global objects for producer/consumer for background metrics publishing
METRICS_QUEUE = Queue.Queue(maxsize=100)
METRICS_THREAD = None

SHUTDOWN_MESSAGE = 'SHUTTING_DOWN'

class _MetricsWorkerThread(threading.Thread):
  """Worker Thread for publishing metrics in the background."""
  
  def __init__(self, tracker):
    threading.Thread.__init__(self, name='metrics-worker')
    global METRICS_QUEUE
    self.tracker = tracker
    self.queue = METRICS_QUEUE
    self.logger = logging.getLogger('graphlab.metrics')
    self.startup_time = time.time()

  def run(self):
    while True:
      try:
        metric = self.queue.get() # block until something received
        self.logger.debug("GOT WORK! %s" % metric)

        if (metric['event_name'] == SHUTDOWN_MESSAGE):
          # shutting down, send total session time and then exit
          self.logger.debug("Got shutdown command, sending session time and exiting")
          now = time.time()
          session_length_secs = now - self.startup_time
          self.tracker._track('session.duration.secs', value=session_length_secs)
          break
      
        self.tracker._track(metric['event_name'], metric['value'], metric['type'], metric['properties'], metric['meta'], metric['send_sys_info'])
                
        self.queue.task_done()
        self.logger.debug("logged '%s' successfully" % metric['event_name'])

      except Queue.Empty:
        # if queue is empty, wait try again
        self.logger.debug("Nothing in Queue, doing nothing")
      except Exception as e:
        self.logger.debug("Exception in processing metrics, %s" % e)

class MetricTracker:
  def __init__(self, mode='UNIT', background_thread=True):
    global METRICS_THREAD, METRICS_QUEUE

    self._valid_modes = ['DEV', 'QA', 'PROD']
    self._tracker = None # librato metrics tracker
    self._mixpanel = None # Mixpanel metrics tracker
    self._url = 'https://metrics.graphlab.com'
    self._usable = False

    self._distinct_id = 'unknown'

    # setup logging
    self.logger = logging.getLogger('graphlab.metrics')

    self._mode = mode

    buffer_size = 5
    offline_buffer_size = 25

    try:
      if self._mode not in self._valid_modes:
        self.logger.info("Using MetricMock instead of real metrics, mode is: %s" % self._mode)
        self._tracker = MetricMock()
        self._mixpanel = MetricMock()
      else:
        self._tracker = librato.connect(CONFIG.librato_user, CONFIG.librato_token)
        self._mixpanel = mixpanel.Mixpanel(CONFIG.mixpanel_user,
                                           OfflineConsumer(max_size=buffer_size,
                                           offline_buffer_size=offline_buffer_size,
                                           events_url=self._url+'/track',
                                           people_url=self._url+'/engage'))
    except Exception, e:
      self.logger.warn("Unexpected exception connecting to Metrics service, disabling metrics, exception %s" % e)
    else:
      self._usable = True

    self._graphlab_version = CONFIG.version
    self._distinct_id = self._get_distinct_id()
    self._source = ("%s-%s" % (self._mode, self._graphlab_version))
    self.logger.debug("Running with metric source: %s" % self._source)

    self._sys_info_set = False

    # background thread for metrics
    self._queue = METRICS_QUEUE
    self._thread = None
    if background_thread:
      self._start_queue_thread()

  def __del__(self):
    self.flush()
    self._stop_queue_thread()

  def track(self, event_name, value=1, type="gauge", properties={}, meta={}, send_sys_info=False):
    """
    Publishes event / metric to metrics providers.

    This method is a facade / proxy, queuing up this metric for a background thread to process.
    """
    try:
      item = dict(event_name=event_name, value=value, type=type, properties=properties, meta=meta, send_sys_info=send_sys_info)
      self._queue.put_nowait(item) # don't wait if Queue is full, just silently ignore
      self.logger.debug("Enqueued event: %s, current queue size: %d" % (event_name, self._queue.qsize()))
    except Queue.Full:
      if not self._thread or not self._thread.is_alive():
        self.logger.debug("Queue is full and background thread is no longer alive, trying to restart")
        self._restart_queue_thread()
      else:
        self.logger.debug("Queue is full, doing nothing.")

    except Exception as e:
      self.logger.debug("Unexpected exception in queueing metrics, %s" % e)
  
  def _stop_queue_thread(self):
    # send the shutting down message, wait for thread to exit
    if self._thread is not None:
      self.track(SHUTDOWN_MESSAGE)
      self._thread.join(2.0)

  def _start_queue_thread(self):
    global METRICS_THREAD
    if (self._thread is None):
      self.logger.debug("Starting background thread")
      self._thread = _MetricsWorkerThread(self)
      METRICS_THREAD = self._thread
      self._thread.daemon = True
      self._thread.start()

  def _restart_queue_thread(self):
    global METRICS_THREAD

    if (self._thread is not None and self._thread.is_alive()):
      self._stop_queue_thread()
      if self._thread.is_alive():
        self.logger.debug("Background thread is still alive, after waiting 2s for it to die. Going to create new thread instead.")

    METRICS_THREAD = None
    del self._thread
    self._thread = None

    self._start_queue_thread()

  def _track(self, event_name, value=1, type="gauge", properties={}, meta={}, send_sys_info=False):
    """
    Internal method to actually send metrics, expected to be called from background thread only.
    """
    if not self._usable:
      return
    the_properties = {}

    if send_sys_info:
      if not self._sys_info_set:
        self._set_sys_info()
      the_properties.update(self._sys_info)

    the_properties.update(properties)

    try:
      # librato
      self._tracker.submit(name=event_name, value=value, type="gauge", source=self._source, attributes=the_properties)

      # since mixpanel cannot send sizes or numbers, just tracks events, bucketize these here
      if value != 1:
          event_name = self._bucketize_mixpanel(event_name, value)
          the_properties['value'] = value

      # mixpanel
      the_properties['source'] = self._source

      self._mixpanel.track(self._distinct_id, event_name, properties=the_properties, meta=meta)

    except Exception as e:
      if (self._mode != 'PROD'):
          self.logger.error("Error adding metrics event to queue., %s" % (e))
          # This is happening either beacuse:
          #   - The user is not connected to the internet
          #   - The server is down (ours or Librato's or Mixpanel's ... whoever)
          #   - Invalid url
      pass

  def flush(self):
    if not self._usable:
      return

    try:
      self.logger.debug("Flushing queued metrics")
      self._tracker.submit()
      self._mixpanel._consumer.flush()
    except:
      pass

  def _bucketize_mixpanel(self, event_name, value):
    """
    Take events that we would like to bucketize and bucketize them before sending to mixpanel

    @param event_name current event name, used to assess if bucketization is required
    @param value value used to decide which bucket for event
    @return event_name if updated then will have bucket appended as suffix, otherwise original returned
    """

    if value == 1:
      return event_name

    if event_name.endswith('col.size'):
        col_buckets = [ 5, 10, 20 ]
        suffix = None
        for bucket in col_buckets:
            if value <= bucket:
                suffix = str(bucket)
                break
        if suffix is None and value > 20:
            suffix = '20+'              
        event_name = '%s.%s' % (event_name, suffix)
    elif event_name.endswith('row.size'):
        row_buckets = [ 100000, 1000000, 10000000, 100000000 ]
        suffix = None
        for bucket in row_buckets:
            if value <= bucket:
                suffix = str(bucket)
                break
        if suffix is None and value > 100000000:
            suffix = '100000000+'
        event_name = '%s.%s' % (event_name, suffix)
    elif event_name.endswith('duration.secs'):
        session_buckets = [ 300, 1800, 3600, 7200 ]
        suffix = None
        for bucket in session_buckets:
            if value <= bucket:
                suffix = str(bucket)
                break
        if suffix is None and value > 7200:
            suffix = '7200+'
        event_name = '%s.%s' % (event_name, suffix)

    return event_name
    

  def _set_sys_info(self):
    # Don't do this if system info has been set
    if self._sys_info_set:
      return

    self._sys_info = {}

    # Get OS-specific info
    self._sys_info['system'] = platform.system()

    if self._sys_info['system'] == 'Linux':
      self._sys_info['os_version'] = self._tup_to_flat_str(platform.linux_distribution())
      self._sys_info['libc_version'] = self._tup_to_flat_str(platform.libc_ver())
    elif self._sys_info['system'] == 'Darwin':
      self._sys_info['os_version'] = self._tup_to_flat_str(platform.mac_ver())
    elif self._sys_info['system'] == 'Windows':
      self._sys_info['os_version'] = self._tup_to_flat_str(platform.win32_ver())
    elif self._sys_info['system'] == 'Java':
      self._sys_info['os_version'] = self._tup_to_flat_str(platform.java_ver())

    # Python specific stuff
    self._sys_info['python_implementation'] = platform.python_implementation()
    self._sys_info['python_version'] = platform.python_version()
    self._sys_info['python_build'] = self._tup_to_flat_str(platform.python_build())

    # Get architecture info
    self._sys_info['architecture'] = self._tup_to_flat_str(platform.architecture())
    self._sys_info['platform'] = platform.platform()
    self._sys_info['num_cpus'] = NUM_CPUS 

    # Get RAM size
    self._sys_info['total_mem'] = TOTAL_PHYMEM

    self._sys_info_set = True

  def _print_sys_info(self):
    pp = pprint.PrettyPrinter(indent=2)
    pp.pprint(self._sys_info)

  def _tup_to_flat_str(self, tup):
    tmp_list = []
    for t in tup:
      if isinstance(t, tuple):
        tmp_str =self._tup_to_flat_str(t)
        tmp_list.append(tmp_str)
      elif isinstance(t, str):
        tmp_list.append(t)
      else:
        # UNEXPECTED! Just don't crash
        try:
          tmp_list.append(str(t))
        except:
          pass

    return " ".join(tmp_list)

  def _get_distinct_id(self):
    if self._distinct_id == 'unknown':
      poss_id = 'unknown'
      id_file_path = "/".join([os.path.abspath(os.path.dirname(__file__)), '..','graphlab',"id"])
      if os.path.isfile(id_file_path):
        try:
          with open(id_file_path, 'r') as f:
            poss_id = f.readline()
        except:
          return "session-" + str(uuid.uuid4())
      else:
        return "session-" + str(uuid.uuid4())
      return poss_id.strip()
    else:
      return self._distinct_id

class OfflineConsumer(mixpanel.BufferedConsumer):
  def __init__(self, max_size, events_url, people_url, offline_buffer_size=25):

    self._offline_buffer_size = offline_buffer_size

    if offline_buffer_size < max_size:
      self._offline_buffer_size = max_size

    super(OfflineConsumer, self).__init__(max_size=max_size, events_url=events_url, people_url=people_url)

  def _flush_endpoint(self, endpoint):
    buf = self._buffers[endpoint]
    while buf:
      batch = buf[:self._max_size]
      batch_json = '[{0}]'.format(','.join(batch))
      try:
        self._consumer.send(endpoint, batch_json)
      except Exception, e:
        if len(buf) < self._offline_buffer_size:
          # Preserve the contents of the buffer
          break
        else:
          # Clear the buffer
          buf = buf[self._offline_buffer_size:]
          break
      buf = buf[self._max_size:]

    self._buffers[endpoint] = buf
