"""
Models that rank items based on their popularity.
"""
import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits.recommender.recommender import RecommenderModel
from graphlab.data_structures.sframe import SFrame as _SFrame
from pandas import DataFrame as _DataFrame


def create(dataset, user, item,
           holdout_probability=0.0,
           verbose=True, plot=False, random_seed=0, **kwargs):
    """
    Create a :class:`~graphlab.recommender.popularity.PopularityModel` that
    ranks items based on their overall popularity. Items are scored by the
    number of times each is seen in the training set. A trained model can be
    used to score (user, item) pairs and make recommendations.

    Parameters
    ----------
    dataset : SFrame/pandas.DataFrame
        The dataset to use for training the model.

    user : string
        The column name of the dataset that corresponds to user id.

    item : string
        The column name of the dataset that corresponds to item id.

    holdout_probability : float, optional
        Proportion of the dataset held out of model training. The held-out
        subset is used to estimate the model's error rate when making
        predictions with new data; the prediction is available in the model's
        training_stats field.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    random_seed : integer, optional
        The random seed used to choose the training and validation
        split if holdout_probability is nonzero.

    Returns
    -------
    out : PopularityModel
        A trained model.

    Examples
    --------
    If given an :class:`~graphlab.SFrame` ``sf`` with columns ``user_id`` and
    ``item_id``, then we can create a
    :class:`~graphlab.popularity.PopularityModel` as follows:

    >>> from graphlab import recommender
    >>> m = recommender.popularity.create(sf, 'user_id', 'item_id')

    With this model object one can make recommendations for the unique users in ``sf``:

    >>> recs = m.recommend(sf)

    The model can be saved to disk as follows:

    >>> m.save(filename)

    For more, see the documentation for
    :class:`~graphlab.recommender.popularity.PopularityModel`.
    """

    _mt._get_metric_tracker().track('toolkit.recsys.popularity.create')

    if not isinstance(dataset, (_DataFrame, _SFrame)):
        raise TypeError('dataset input must be a pandas.DataFrame or SFrame')

    if type(dataset) != _SFrame:
        dataset = _SFrame(dataset)

    if plot is True:
        print "The plot functionality for popularity is not yet implemented."
        plot = False

    opts = {'model_name': 'item_counts'}
    response = _graphlab.toolkits.main.run("recsys_init", opts)
    model_proxy = response['model']

    opts = {'user_column': user,
            'item_column': item,
            'training_data': dataset,
            'model': model_proxy,
            'random_seed' : random_seed,
            'holdout_probability': holdout_probability}
    opts.update(kwargs)
    response = _graphlab.toolkits.main.run("recsys_train", opts, verbose, plot)

    return PopularityModel(model_proxy)


class PopularityModel(RecommenderModel):
    """
    Item popularity model for the GraphLab recommender system. Each user's
    recommendations are the overall most popular items in the data.
    """

    def __init__(self, model_proxy):
        self.__proxy__ = model_proxy

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return PopularityModel(model_proxy)
        return model_wrapper
