
from graphlab.data_structures.graph import Graph
import graphlab.toolkits.main as main
from graphlab.toolkits.model import Model


class ConnectedComponentModel(Model):
    """
    A Model object containing a componentid for each vertex and the total
    number of weakly connected component in the graph.
    """
    def __init__(self, input_params, model):
        self.__proxy__ = model
        self._input = input_params
        self._fields = ['num_of_components', 'runtime', 'componentid', 'graph',
            'component_size']

    def list_fields(self):
        """Return the list of fields in the model."""

        return self._fields

    def get(self, field):
        """Return the value for the queried field."""

        if (field == 'componentid'):
            return Graph(_proxy=self.__proxy__.get('__graph__')).select_fields(['componentid']).get_vertices()
        elif (field == 'graph'):
            return Graph(_proxy=self.__proxy__.get('__graph__'))
        else:
            return self.__proxy__.get(field)

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""

        ret = {'hyperparams': self._input}
        for key in ['runtime', 'num_of_components']:
            ret[key] = self.get(key)
        return ret

    def _get_wrapper(self):
        return lambda m: ConnectedComponentModel(self._input, m)


def create(graph, verbose=False, plot=False):
    """
    Compute the number of weakly connected components in the graph. Return a
    model object with total number of weakly connected components as well as the
    component ID for each vertex in the graph.

    Parameters
    ----------
    graph : Graph
        The graph on which to compute the triangle counts.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : Model
        Model object that contains the componentid for each vertex.
    
    Examples
    --------
    If given an :class:`~graphlab.Graph` ``g``, we can create
    a :class:`~graphlab.connected_components.ConnectedComponentModel` as
    follows:

    >>> cc = connected_components.create(g)
    >>> print cc.summary()
    >>> print cc.list_fields()

    We can obtain the ``component id`` corresponding to each vertex in the
    graph ``g`` as follows: 

    >>> cc_ids = cc.get('componentid')  # SFrame
    
    We can obtain a graph with additional information about the ``component
    id`` corresponding to each vertex as follows:
    
    >>> cc_graph = cc.get('graph')      # GraphLab Graph

    """
    if not isinstance(graph, Graph):
        raise TypeError('graph input must be a Graph object.')

    if plot is True:
        print "The plot functionality for connected components is not yet " + \
            "implemented."
        plot = False

    if verbose is True:
        print "Starting connected components computation."
        verbose = 0

    params = main.run('connected_component', {'graph': graph.__proxy__},
                      verbose, plot)
    return ConnectedComponentModel({}, params['model'])
