
from graphlab.data_structures.graph import Graph
import graphlab.toolkits.main as main
from graphlab.toolkits.model import Model


class GraphColoringModel(Model):
    """
    A Model object containing a color ID for each vertex and the total number
    of colors used.
    """
    def __init__(self, input_params, model):
        self.__proxy__ = model
        self._input = input_params
        self._fields = ['num_colors', 'runtime', 'colorid', 'graph']

    def list_fields(self):
        """Return the list of fields in the model."""

        return self._fields

    def get(self, field):
        """Return the value for the queried field."""

        if (field == 'colorid'):
            return Graph(_proxy=self.__proxy__.get('__graph__')).select_fields(['colorid']).get_vertices()
        elif (field == 'graph'):
            return Graph(_proxy=self.__proxy__.get('__graph__'))
        else:
            return self.__proxy__.get(field)

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""

        ret = {'hyperparams': self._input}
        for key in ['runtime', 'num_colors']:
            ret[key] = self.get(key)
        return ret

    def _get_wrapper(self):
        return lambda m: GraphColoringModel(self._input, m)


def create(graph, verbose=False, plot=False):
    """
    Compute the graph coloring. Assign a color to each vertex such that no
    adjacent vertices have the same color. Return a model object with total
    number of colors used as well as the color ID for each vertex in the graph.
    This algorithm is greedy and is not guaranteed to find the **minimum** graph
    coloring. It is also not deterministic, so successive runs may return
    different answers.

    Parameters
    ----------
    graph : Graph
        The graph on which to compute the coloring.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : Model
        Model object that contains the color id for each vertex.

    References
    ----------
    - `Wikipedia - graph coloring <http://en.wikipedia.org/wiki/Graph_coloring>`_
    
    Examples
    --------
    If given an :class:`~graphlab.Graph` ``g``, we can create
    a :class:`~graphlab.graph_coloring.GraphColoringModel` as follows:

    >>> gc = graph_coloring.create(g)

    We can obtain the ``color id`` corresponding to each vertex in the graph ``g``
    as follows: 

    >>> color_id = gc.get('colorid')  # SFrame
    
    We can obtain the total number of colors required to color the graph ``g``
    as follows:

    >>> num_colors = gc.get('num_colors')
    """
    if not isinstance(graph, Graph):
        raise TypeError('graph input must be a Graph object.')

    if plot is True:
        print "The plot functionality for graph coloring is not yet implemented."
        plot = False

    if verbose is True:
        print "Starting graph coloring computation."
        verbose = 0

    params = main.run('graph_coloring', {'graph': graph.__proxy__}, verbose,
                      plot)

    return GraphColoringModel({}, params['model'])
