
from graphlab.data_structures.graph import Graph
import graphlab.toolkits.main as main
from graphlab.toolkits.model import Model


class TriangleCountingModel(Model):
    """
    A Model object containing the traingle_count for each vertex, and the total
    number of triangles.
    """
    def __init__(self, input_params, model):
        self.__proxy__ = model
        self._input = input_params
        self._fields = ['num_triangles', 'runtime', 'triangle_count', 'graph']

    def list_fields(self):
        """Return the list of fields in the model."""

        return self._fields

    def get(self, field):
        """Return the value for the queried field."""

        if (field == 'triangle_count'):
            return Graph(_proxy=self.__proxy__.get('__graph__')).select_fields(['triangle_count']).get_vertices()
        elif (field == 'graph'):
            return Graph(_proxy=self.__proxy__.get('__graph__'))
        else:
            return self.__proxy__.get(field)

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""

        ret = {'hyperparams': self._input}
        for key in ['runtime', 'num_triangles']:
            ret[key] = self.get(key)
        return ret

    def _get_wrapper(self):
        return lambda m: TriangleCountingModel(self._input, m)


def create(graph, verbose=False, plot=False):
    """
    Compute the number of triangles each vertex belongs to, ignoring edge
    directions. A triangle is a complete subgraph with only three vertices.
    Return a model object with total number of triangles as well as the triangle
    counts for each vertex in the graph.

    Parameters
    ----------
    graph : Graph
        The graph on which to compute triangle counts.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : Model
        A Model object that contains the total and individual triangle counts.
    
    Examples
    --------
    If given an :class:`~graphlab.Graph` ``g``, we can create
    a :class:`~graphlab.traingle_counting.TriangleCountingModel` as follows:

    >>> tc = triangle_counting.create(g)

    We can obtain the number of triangles that each vertex in the graph ``g``
    is present in: 

    >>> tc_out = tc.get('triangle_count')  # SFrame
    
    """
    if not isinstance(graph, Graph):
        raise TypeError('graph input must be a Graph object.')

    if plot is True:
        print "The plot functionality for triangle counting is not yet implemented."
        plot = False

    if verbose is True:
        print "Starting triangle counting computation."
        verbose = 0

    params = main.run('triangle_counting', {'graph': graph.__proxy__}, verbose, plot)
    return TriangleCountingModel({}, params['model'])
