
from graphlab.data_structures.sgraph import SGraph
import graphlab.toolkits.main as main
from graphlab.toolkits.model import Model


class KcoreModel(Model):
    """
    A Model object containing a coreid for each vertex, and the total number of
    cores in the graph.

    An instance of this model can be created using :func:`graphlab.kcore.create`.
    Do NOT construct the model directly.
    """
    def __init__(self, input_params, model):
        '''__init__(self)'''
        self.__proxy__ = model
        self._input = input_params
        self._fields = ['coreid', 'runtime', 'max_core', 'graph']

    def list_fields(self):
        """
        List of fields stored in the model. Each of these fields can be queried
        using the ``get`` function.

        Returns
        -------
        out : list
            A list of fields that can be queried using the ``get`` method.
        """
        return self._fields

    def get(self, field):
        """
        Get the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained programmatically using the
        :func:`~graphlab.graph_analytics.KcoreModel.list_fields`
        method.

        Each of these fields can be queried in one of two ways:

        +-----------------+----------------------------------------------------+
        |      Field      | Description                                        |
        +=================+====================================================+
        | coreid          | SFrame with the core ID of each vertex             |
        +-----------------+----------------------------------------------------+
        | graph           | Input graph with core ID as a vertex property      |
        +-----------------+----------------------------------------------------+

        >>> out = m['field']
        >>> out = m.get('field')  # equivalent to previous line

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out : [various]
            The current value of the requested field.

       """

        if (field == 'coreid'):
            return SGraph(_proxy=self.__proxy__.get('__graph__')).select_fields(['coreid']).get_vertices()
        elif (field == 'graph'):
            return SGraph(_proxy=self.__proxy__.get('__graph__'))
        else:
            return self.__proxy__.get(field)

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""

        ret = {'hyperparams': self._input}
        for key in ['runtime', 'max_core']:
            ret[key] = self.get(key)
        return ret

    def _get_wrapper(self):
        return lambda m: KcoreModel(self._input, m)


def create(graph, kmin=1, kmax=10, verbose=True, plot=False):
    """
    Compute the K-core decomposition of the graph. Return a model object with
    total number of cores as well as the core id for each vertex in the graph.

    Parameters
    ----------
    graph : SGraph
        The graph on which to compute the k-core decomposition.

    kmin : int, optional
        Minimun core id. Vertices having smaller core id than `kmin` will be
        assigned with coreid = `kmin`.

    kmax : int, optional
        Maximun core id. Vertices having larger core id than `kmax` will be
        assigned with coreid=`kmax`.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : Model
        A Model object that contains the coreid for each vertex.

    References
    ----------
    - Alvarez-Hamelin, J.I., et al. (2005) `K-Core Decomposition: A Tool for the
      Visualization of Large Networks <http://arxiv.org/abs/cs/0504107>`_.

    Examples
    --------
    If given an :class:`~graphlab.SGraph` ``g``, we can create
    a :class:`~graphlab.kcore.KcoreModel` as follows:

    >>> kc = kcore.create(g)

    We can obtain the ``core id`` corresponding to each vertex in the graph
    ``g`` using:

    >>> kcore_id = kc['coreid']     # SFrame
    """
    if not isinstance(graph, SGraph):
        raise TypeError('graph input must be a SGraph object.')

    opts = {'graph': graph.__proxy__, 'kmin': kmin, 'kmax': kmax}
    params = main.run('kcore', opts, verbose, plot)

    return KcoreModel({'kmin': kmin, 'kmax': kmax}, params['model'])
