"""
Methods for performing matrix factorization.  See
graphlab.matrix_factorization.create for additional documentation.
"""

import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits.recommender.recommender import RecommenderModel
from graphlab.data_structures.sframe import SFrame as _SFrame
import time
from pandas import DataFrame as _DataFrame
import logging

def create(dataset,
           user_column='user_id', item_column='item_id',
           target_column='target',
           user_data=None, item_data=None,
           n_factors=10, regularization=0.01, nmf=False,
           max_iterations=50,
           verbose=True, plot=False, random_seed=None, **kwargs):
    """
    Create a :class:`~graphlab.recommender.matrix_factorization.MatrixFactorizationModel`.
    A trained model can be used to score (user, item) pairs and make recommendations.

    This is an implementation of basic Alternating Least Squares with
    user bias, item bias and a global bias term.

    NOTE: This will be *deprecated* soon. Please use graphlab.recommender.create instead.

    References
    ----------
    See this paper for details about the model: Koren, Yehuda, Robert Bell and
    Chris Volinsky. "Matrix Factorization Techniques for Recommender Systems."
    Computer Volume: 42, Issue: 8 (2009): 30-37

    Parameters
    ----------
    dataset : SFrame
        The dataset to use for training the model.

    user_column : string
        The column name of the dataset that corresponds to user id.

    item_column : string
        The column name of the dataset that corresponds to item id.

    target_column : string
        The model will be trained to predict this column of the data set.

    user_data : SFrame
        Side information for the users.  This SFrame must have a column named
        with the  user column name given above. It provides any amount of
        additional user-specific information.

    item_data: SFrame
        Side information for the items. This SFrame must have a column named
        with the item column name given above. It provides any amount of
        additional item-specific information.

    n_factors : int, optional
        Number of latent factors to use for describing each item (default is
        10).

    regularization : float, optional

    Regularization weight (default is 0.01)

    nmf : boolean, optional
        Non-negative matrix factorization flag. If True, the model is constrained
        to use only non-negative matrices (default is False).

    max_iterations : int, optional
        Number of optimization iterations to run.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    random_seed : integer, optional
        The random seed used to choose the starting point of the optimization.

    Returns
    -------
    out : MatrixFactorizationModel
        A trained model.

    Examples
    --------
    If given an :class:`~graphlab.SFrame` ``sf`` with columns ``user_id`` and ``item_id`` and
    a column we would like to predict called ``target``, then we can
    create a :class:`graphlab.matrix_factorization.MatrixFactorizationModel` as follows:

    >>> from graphlab import recommender
    >>> m = recommender.matrix_factorization.create(sf, 'user_id', 'item_id', target='target')

    With this model object one can make recommendations for the unique users in ``sf``:

    >>> recs = m.recommend(sf)

    The model can be saved to disk as follows:

    >>> m.save(filename)

    For more, see the documentation for :class:`~graphlab.recommender.matrix_factorization.MatrixFactorizationModel`.
    """

    _mt._get_metric_tracker().track('toolkit.recsys.matrix_factorization.create')

    logging.warning("matrix_factorization.create will be deprecated soon. Please switch to recommender.create")

    if not isinstance(dataset, (_DataFrame, _SFrame)):
        raise TypeError('dataset input must be a pandas.DataFrame or SFrame')

    if type(dataset) != _SFrame:
        dataset = _SFrame(dataset)

    if not random_seed:
        random_seed = int(time.time())

    method_options = {'model_name': 'matrix_factorization',
                      'n_factors': n_factors,
                      'regularization': regularization,
                      'nmf': nmf,
                      'max_iterations': max_iterations}
    method_options.update(kwargs)

    m = _graphlab.recommender.create(dataset,
                                     user_column=user_column,
                                     item_column=item_column,
                                     target_column=target_column,
                                     user_data=user_data, item_data=item_data,
                                     verbose=verbose,
                                     method='matrix_factorization',
                                     **method_options)

    return m


class MatrixFactorizationModel(RecommenderModel):
    """
    A matrix factorization model is able to learn a set of parameters for each user and for each item, scoring a (user, item) pair based on the similarity of the user and the item.
    In other words, the method will compose a set of user-item ratings (when represented as a matrix) into a
    number of latent factors for each item and corresponding weights for each
    user.

    An instance of this model can be created using
    :func:`create(..., method='matrix_factorization') <graphlab.recommender.create>`.
    Do NOT construct the model directly.

    NOTE: This will be deprecated soon. Please use
    :func:`create(..., method='matrix_factorization') <graphlab.recommender.create>`
    instead.

    """

    def __init__(self, model_proxy):
        '''__init__(self)'''
        self.__proxy__ = model_proxy

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return MatrixFactorizationModel(model_proxy)
        return model_wrapper
