r"""
The GraphLab regression toolkit includes linear and logistic regression models
that can be trained to make predictions about new data.

GraphLab regression models accept training and prediction data in the form of an
:class:`~graphlab.SFrame`. Training datasets should contain a column for the
target variable and one or more columns representing feature variables. For
predictions on new data, the SFrame need not contain the target column.

The :py:class:`~graphlab.linear_regression.LinearRegressionModel` model
predicts a scalar target variable as a linear function of one or more
feature variables. 

.. sourcecode:: python

    >>> from graphlab import linear_regression, SFrame
    >>> data = SFrame('train-medical-data.csv')
    >>> m = linear_regression.create(data, target='bloodpressure',
                                     features=['age', 'weight', 'height'])
    >>> test_data = SFrame('test-medical-data.csv')
    >>> coefficients = m['coefficients']     # an SFrame
    >>> predictions = m.predict(test_data)   # an SArray
    >>> results = m.evaluate(test_data)      # a dictionary


The :class:`~graphlab.logistic_regression.LogisticRegressionModel` uses a binary
target variable :math:`y` instead of a scalar. For each observation, the
probability that :math:`y=1` (instead of 0) is modeled as the logistic
function of a linear combination of the feature values. Logistic regression
predictions may take the form of the linear combination (also known as the
margin), the probability that :math:`y = 1`, or a binary class value, which is
obtained by thresholding the probability at 0.5.

.. sourcecode:: python

    >>> from graphlab import logistic_regression, SFrame
    >>> data = SFrame('train-admissions-data.csv')
    >>> m = logistic_regression.create(data, target='admission',
                                       features=['GPA', 'SAT-score', 'essay-score'])
    >>> test_data = SFrame('test-admissions-data.csv')
    >>> coefficients = m['coefficients']
    >>> predictions = m.predict(test_data, output_type='probability')
    >>> results = m.evaluate(test_data)
"""

from graphlab.toolkits.regression import linear_regression
from graphlab.toolkits.regression import logistic_regression
