"""
Methods for creating and using a logistic regression model.
"""
import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits.regression.regression import RegressionModel
from graphlab.data_structures.sframe import SArray as _SArray
from graphlab.data_structures.sframe import SFrame as _SFrame
from graphlab import vowpal_wabbit as _vw
from pandas import DataFrame as _DataFrame
import math
from graphlab.toolkits.main import ToolkitError

DEFAULT_SOLVER_OPTIONS = {
'convergence_threshold': 1e-2,
'step_size': 1.0,
'mini_batch_size': 1,
'max_iterations': 10}


def create(dataset, target, features=None, L2_penalty=0.01, L1_penalty=0.0,
           solver='auto', solver_options=None, verbose=True):
    """
    Create a :class:`~graphlab.logistic_regression.LogisticRegressionModel` to
    predict the class of a binary target variable based on a model of class
    probability as a logistic function of a linear combination of the features.
    In addition to standard numeric and categorical types, features can also be
    extracted automatically from list- or dictionary-type SFrame columns.

    This model can be regularized with an L1 penalty, an L2 penalty, or both. By
    default this model has an L2 regularization weight of 0.01.

    Parameters
    ----------
    dataset : pandas.DataFrame/SFrame
        Dataset for training the model.

    target : string
        Name of the column containing the target variable. The values in this
        column must be 0 or 1, of integer type.

    features : list[string], optional
        Names of the columns containing features. 'None' (the default) indicates
        that all columns except the target variable should be used as features.

        - *Numeric*: values of numeric type integer or float.

        - *Categorical*: values of type string.

        - *List*: list of numeric (integer or float) values. Each list element
          is treated as a separate feature in the model.

        - *Dictionary*: key-value pairs with numeric (integer or float) values.
          Each key of a dictionary is treated as a categorical variable in the
          model.

    L2_penalty : float, optional
        Weight on L2 regularization of the model. The larger this weight, the
        more the model coefficients shrink toward 0. This introduces bias into
        the model but decreases variance, potentially leading to better
        predictions. The default value is 0.01; setting this parameter to 0
        corresponds to unregularized logistic regression. See the ridge
        regression reference for more detail.

    L1_penalty : float, optional
        Weight on L1 regularization of the model. Like the L2 penalty, the
        higher the L1 penalty, the more the estimated coefficients shrink toward
        0. The L1 penalty, however, completely zeros out sufficiently small
        coefficients, automatically indicating features that are not useful
        for the model. The default weight of 0 prevents any features from
        being discarded. See the LASSO regression reference for more detail.

    solver : string, optional
        Name of the solver to be used to solve the regression. See the
        references for more detail on each solver. Available solvers are:

        - *auto (default)*: automatically chooses the best solver for the data
          and model parameters.
        - *newton*: Newton-Raphson
        - *lbfgs*: limited memory BFGS
        - *vw*: Vowpal Wabbit
        - *fista*: accelerated gradient descent

        For this model, the Newton-Raphson method is equivalent to the
        iteratively re-weighted least squares algorithm. If the L1_penalty is
        greater than 0, use either the 'fista' or 'vw' solver.

    solver_options : dict, optional
        Solver options. The options and their default values are as follows:

        +-----------------------+---------+-----------------------------------------+
        |      Option           | Default |        Description                      |
        +=======================+=========+=========================================+
        | convergence_threshold |    1e-2 | Desired training accuracy               |
        +-----------------------+---------+-----------------------------------------+
        | lbfgs_memory_level    |     3   | Number of updates to store (lbfgs only) |
        +-----------------------+---------+-----------------------------------------+
        | max_iterations        |     10  | Max number of solver iterations         |
        +-----------------------+---------+-----------------------------------------+
        | step_size             |    1.0  | Initial solver step size                |
        +-----------------------+---------+-----------------------------------------+

    verbose : bool, optional
        If True, print progress updates.

    Returns
    -------
    out : LogisticRegressionModel
        A trained model of type
        :class:`~graphlab.logistic_regression.LogisticRegressionModel`.

    Notes
    -----
    - Categorical variables are encoded by creating dummy variables. For a
      variable with :math:`K` categories, the encoding creates :math:`K-1` dummy
      variables, while the first category encountered in the data is used as the
      baseline.

    - For prediction and evaluation of logistic regression models with
      categorical variables, test datasets cannot contain categories that were
      not present during train time.

    - For prediction and evaluation of logistic regression models with sparse
      dictionary inputs, new keys/columns that were not seen during training
      are silently ignored.

    - L2 and L1 regularization typically work best if the features are
      standardized before model training by subtracting the mean of column and
      dividing by the standard deviation. The regression model does *not* do
      this automatically; use SFrame and SArray methods to transform the data
      prior to training.

    - If the 'vw' solver is used, the model returns only predictions; it does
      not allow inspection of estimated effect sizes or standard errors.

    - Any 'None' values in the data will result in an error being thrown.

    - A constant term is automatically added for the model intercept. This term
      is not regularized.

    Examples
    --------
    *Training*

    Given an :class:`~graphlab.SFrame` ``sf``, a list of feature columns
    [``feature_1`` ... ``feature_K``], and a target column ``target`` with 0 and
    1 values, create a
    :class:`~graphlab.logistic_regression.LogisticRegressionModel` as follows:

    >>> m = logistic_regression.create(sf, 'target')

    By default all columns of ``sf`` except the target are used as features, but
    specific feature columns can be specified manually.

    >>> m = logistic_regression.create(sf, 'target', ['feature_1', 'feature_2'])

    The L2 and L1 regularization weights are set to 0.01 and 0.0 by default. For
    stronger regularization raise one or both of these weights.

    >>> m = logistic_regression.create(sf, 'target', L2_penalty=0.5,
                                       L1_penalty=1.)

    *Model Querying*

    Model attributes can be retrieved with either bracket syntax or the
    :func:`~graphlab.logistic_regression.LogisticRegressionModel.get` method. The
    set of queryable fields is described in the ``get`` documentation, and
    can be obtained programmatically with the
    :func:`~graphlab.logistic_regression.LogisticRegressionModel.list_fields`
    method. The model coefficients are stored in an SFrame.

    >>> m.list_fields()
    >>> coef = m['coefficients']  # an SFrame
    >>> coef = m.get('coefficients')  # equivalent to previous line

    The :func:`~graphlab.logistic_regression.LogisticRegressionModel.summary`
    and
    :func:`~graphlab.logistic_regression.LogisticRegressionModel.training_stats`
    methods provide shortcuts to many of the model attributes.

    >>> print m.summary()
    >>> m.training_stats()

    *Prediction and Evaluation*

    With the trained model we can make predictions for new data in SFrame
    ``sf_new`` and compare these predictions to actual target values, if these
    values are known. For both of these methods, the new SFrame must include
    columns with the same names as the training features; the ``evaluate``
    method requires ``sf_new`` to have the same ``target`` column name as well.

    Predictions can be generated as class labels (0 or 1), probabilities that
    the target value is 1 (rather than 0), or margins (i.e. the distance of the
    observations from the hyperplane separating the classes). By default, the
    predict method returns probabilities.

    >>> pred_probs = m.predict(sf_new)  # class probabilities P(Y=1)
    >>> pred_margin = m.predict(sf_new, output_type='margin')
    >>> pred_class = m.predict(sf_new, output_type='class')

    *Save and Load*

    The model can be saved to disk and loaded at a later time.

    >>> m.save("mymodel")
    >>> m2 = graphlab.load_model("mymodel")

    For more detail, see the documentation for individual methods of the
    :class:`~graphlab.logistic_regression.LogisticRegressionModel`.

    References
    ----------
    - `Wikipedia - logistic regression
      <http://en.wikipedia.org/wiki/Logistic_regression>`_

    - Hoerl, A.E. and Kennard, R.W. (1970) `Ridge regression: Biased Estimation
      for Nonorthogonal Problems
      <http://amstat.tandfonline.com/doi/abs/10.1080/00401706.1970.10488634>`_.
      Technometrics 12(1) pp.55-67

    - Tibshirani, R. (1996) `Regression Shrinkage and Selection via the Lasso <h
      ttp://www.jstor.org/discover/10.2307/2346178?uid=3739256&uid=2&uid=4&sid=2
      1104169934983>`_. Journal of the Royal Statistical Society. Series B
      (Methodological) 58(1) pp.267-288.

    - Zhu, C., et al. (1997) `Algorithm 778: L-BFGS-B: Fortran subroutines for
      large-scale bound-constrained optimization
      <http://dl.acm.org/citation.cfm?id=279236>`_. ACM Transactions on
      Mathematical Software 23(4) pp.550-560.

    - Beck, A. and Teboulle, M. (2009) `A Fast Iterative Shrinkage-Thresholding
      Algorithm for Linear Inverse Problems
      <http://epubs.siam.org/doi/abs/10.1137/080716542>`_. SIAM Journal on
      Imaging Sciences 2(1) pp.183-202.

    - `Vowpal Wabbit website <http://hunch.net/~vw/>`_

    - `Vowpal Wabbit on Github
      <https://github.com/JohnLangford/vowpal_wabbit/wiki>`_
    """

    model_name = "regression_logistic_regression"
    _mt._get_metric_tracker().track(
        'toolkit.regression.logistic_regression.create')


    ## Check data matrix type and convert to SFrame
    if not isinstance(dataset, (_DataFrame, _SFrame)):
        raise TypeError('Dataset input must be an SFrame.')

    if type(dataset) != _SFrame:
        dataset = _SFrame(dataset)


    # Make the solver name and all solver options lower case. Also keeps a
    # separate copy of solver_options to prevent changes.
    solver = solver.lower()

    if solver_options is not None:
        _solver_options = {k.lower(): v for k,v in solver_options.items()}
    else:
        _solver_options = {}

    # Extract the target vector and check the type and values
    sf_target = dataset.select_columns([target])

    target_type = sf_target[target].dtype()
    if not target_type == int:
        raise TypeError("Target column must be type int")

    # Check that target values are all +1 or 0
    zero_or_one = (sf_target[target] == 0) + (sf_target[target] == 1)
    if not all(zero_or_one):
        raise ValueError("The target column for logistic regression" + \
            " must contain only 0 and 1 values.")

    # Extract the features
    if features is None:
        features = dataset.column_names()
        features.remove(target)

    if not hasattr(features, '__iter__'):
        raise TypeError("Input 'features' must be an iterable")

    if not all([isinstance(x, str) for x in features]):
        raise TypeError("Invalid type: predictor names must be str")

    sf_features = dataset.select_columns(features)


    # Set up options dictionary and initialize the model
    opts = {}
    opts.update(_solver_options)
    opts.update({'target'     : sf_target,
                'features'    : sf_features,
                'model_name'  : model_name,
                'solver'      : solver,
                'l2_penalty'  : L2_penalty,
                'l1_penalty'  : L1_penalty})


    # Model checking
    ret = _graphlab.toolkits.main.run("regression_train_init", opts)
    opts.update(ret)
    model = LogisticRegressionModel(ret['model'])


    if solver == 'vw' or solver == 'vowpal_wabbit':

        # Make the SFrame into VW format
        required_columns = features + [target]
        sf = dataset.select_columns(required_columns)

        # Translate the 0/1 targets to -1/+1 targets
        sf[target] = sf[target] * 2 - 1

        # Append options
        vw_options = {}
        for key in ['max_iterations', 'step_size']:
            if key in _solver_options:
                vw_options[key] = _solver_options[key]
            else:
                vw_options[key] = DEFAULT_SOLVER_OPTIONS[key]

        # Call VW
        # Note: VW checks that the target columns has +1 or -1
        vw_model = _vw.create(sf, target,
                              l1_penalty = L1_penalty,
                              l2_penalty = L2_penalty,
                              loss_function ='logistic',
                              step_size = vw_options['step_size'],
                              verbose = verbose,
                              max_iterations = vw_options['max_iterations'])

        # Save the 'solver' option so that the regression toolkit
        # can query for it when testing which solver is active.
        vw_model._set('solver', 'vw')

        # Use the proxy object for VW
        model = LogisticRegressionModel(vw_model.__proxy__, use_vw=True)

    else:  # all other solvers beside VW

        try:
            ret = _graphlab.toolkits.main.run("regression_train", opts, verbose)
        except:
            raise ValueError("Model failed to train")
        else:
            model_proxy = ret['model']
            model = LogisticRegressionModel(model_proxy)

    model.summary()
    return model


class LogisticRegressionModel(RegressionModel):
    """
    Logistic regression models a binary target variable as a function of several
    feature variables.

    An instance of this model can be created using :func:`graphlab.logistic_regression.create`.
    Do NOT construct the model directly.
    """
    def __init__(self, model_proxy, use_vw=False):
        '''__init__(self)'''
        self.__proxy__ = model_proxy
        self.__name__ = "regression_logistic_regression"
        self.use_vw = use_vw

    def _get_wrapper(self):

        def vw_model_wrapper(model_proxy):
            vw_model = _vw.VWModel(model_proxy)
            return LogisticRegressionModel(model_proxy, use_vw=True)

        def model_wrapper(model_proxy):
            return LogisticRegressionModel(model_proxy)

        if self.get('solver') == 'vw':
            return vw_model_wrapper
        else:
            return model_wrapper

    def __str__(self):
        """
        Return a string description of the model to the ``print`` method.

        Returns
        -------
        out : string
            A description of the model.
        """
        return self.__repr__()

    def __repr__(self):
        """
        Print a string description of the model, when the model name is entered
        in the terminal.
        """

        solver = self.get('solver')
        width = 20
        key_str = "{:<{}}: {}"
        model_fields = [
            ("L1 penalty", 'l1_penalty'),
            ("L2 penalty", 'l2_penalty'),
            ("Examples", 'num_examples'),
            ("Features", 'num_features'),
            ("Coefficients", 'num_coefficients')]

        solver_fields = [
            ("Solver", 'solver'),
            ("Solver iterations", 'train_iters'),
            ("Solver status", 'solver_status'),
            ("Training time (sec)", 'train_time')]

        train_fields = [
            ("Log-likelihood", 'train_loss')]

        ret = []
        ret.append(key_str.format("Class", width, self.__class__.__name__))

        if solver == 'vw':
            m = _vw.VWModel(self.__proxy__)
            ret.append(m.__repr__())

        else:
            for tranche_fields in [model_fields, solver_fields, train_fields]:
                for k, v in tranche_fields:
                    value = self.get(v)
                    if isinstance(value, float):
                        try:
                            value = round(value, 4)
                        except:
                            pass
                    ret.append(key_str.format(k, width, value))
                ret.append("")

        return '\n'.join(ret)

    def get(self, field):
        """
        Return the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained programmatically with the
        :func:`~graphlab.logistic_regression.LogisticRegressionModel.list_fields`
        method. Note that some options are not available if the model is trained
        with the ``vw`` solver.

        Each of these fields can be queried in one of two ways:

        >>> out = m['field']
        >>> out = m.get('field')  # equivalent to previous line

        +-----------------------+----------------------------------------------+
        |      Field            | Description                                  |
        +=======================+==============================================+
        | coefficients          | Regression coefficients (non-'vw' only)      |
        +-----------------------+----------------------------------------------+
        | convergence_threshold | Desired solver accuracy                      |
        +-----------------------+----------------------------------------------+
        | features              | Feature column names                         |
        +-----------------------+----------------------------------------------+
        | l1_penalty            | L1 regularization weight                     |
        +-----------------------+----------------------------------------------+
        | l2_penalty            | L2 regularization weight                     |
        +-----------------------+----------------------------------------------+
        | lbfgs_memory_level    | LBFGS memory level                           |
        +-----------------------+----------------------------------------------+
        | max_iterations        | Maximum number of solver iterations          |
        +-----------------------+----------------------------------------------+
        | num_coefficients      | Number of coefficients in the model          |
        +-----------------------+----------------------------------------------+
        | num_examples          | Number of examples used for training         |
        +-----------------------+----------------------------------------------+
        | num_features          | Number of dataset columns used for training  |
        +-----------------------+----------------------------------------------+
        | solver                | Type of solver                               |
        +-----------------------+----------------------------------------------+
        | solver_status         | Solver status after training                 |
        +-----------------------+----------------------------------------------+
        | step_size             | Initial step size for the solver             |
        +-----------------------+----------------------------------------------+
        | target                | Target column name                           |
        +-----------------------+----------------------------------------------+
        | train_iters           | Number of solver iterations                  |
        +-----------------------+----------------------------------------------+
        | train_loss            | Maximized Log-likelihood                     |
        +-----------------------+----------------------------------------------+
        | train_time            | Training time (excludes preprocessing)       |
        +-----------------------+----------------------------------------------+

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out
            Value of the requested fields.
        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.get')

        # Note: For VW as the solver, map the options back to the options
        # stored in VW. This lets us have control over what the options
        # should look like when VW is the solver.
        if self.use_vw:
            if field == 'coefficients':
                raise ValueError("Models trained with 'vw' as the" + \
                                "solver cannot provide coefficients.")
            vw_option_map = {'solver'         : 'solver',
                             'target'         : 'target_column',
                             'step_size'      : 'step_size',
                             'l1_penalty'     : 'l1_penalty',
                             'l2_penalty'     : 'l2_penalty',
                             'max_iterations'      : 'max_iterations',
                             'train_iters'    : 'max_iterations',
                             'train_time'     : 'elapsed_time',
                             'train_loss'     : 'loss_function',
                             'train_accuracy' : 'train_accuracy'}

            # Only return options in the map
            if field not in vw_option_map:
                raise ValueError, "Field '{}' does not exist. Use".format(field) + \
                    " list_fields() for a list of fields that can be queried."
            return _vw.VWModel(self.__proxy__).get(vw_option_map[field])

        else:
            opts = {'model': self.__proxy__, 'model_name': self.__name__,
                    'field': field}
            response = _graphlab.toolkits.main.run('regression_get_value', opts)

            # Coefficients returns a unity SFrame. Cast to an SFrame.
            if field == 'coefficients':
                return _SFrame(None, _proxy=response['value'])
            else:
                return response['value']

    def summary(self):
        """
        Display a summary of the model, including coefficients (if applicable),
        training options, and training statistics. Use the
        :func:`~graphlab.logistic_regression.LogisticRegressionModel.get` method
        to retrieve these values programmatically, or to see more detail about
        the queryable fields.
        """
        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.summary')
        super(LogisticRegressionModel, self).summary()

    def get_default_options(self):
        """
        Return a dictionary with the model's default options.

        Returns
        -------
        out : dict
            Dictionary with model default options.
        """
        _mt._get_metric_tracker().track(
           'toolkit.regression.logistic_regression.get_default_options')
        return super(LogisticRegressionModel, self).get_default_options()

    def get_options(self):
        """
        Return a dictionary with the options used to define and train the model.

        Returns
        -------
        out : dict
            Dictionary with options used to define and train the model.
        """
        _mt.get_metric_tracker().track(
           'toolkit.regression.logistic_regression.get_options')
        return super(LogisticRegressionModel, self).get_options()

    def predict(self, dataset, output_type='probability'):
        """
        Return predictions for ``dataset``, using the trained logistic
        regression model. Predictions can be generated as class labels (0 or 1),
        probabilities that the target value is 1 (rather than 0), or margins
        (i.e. the distance of the observations from the hyperplane separating
        the classes). By default, the predict method returns probabilities.

        For each new example in ``dataset``, the margin---also known as the
        linear predictor---is the inner product of the example and the model
        coefficients. The probability is obtained by passing the margin through
        the logistic function. Predicted classes are obtained by thresholding
        the predicted probabilities at 0.5.

        Parameters
        ----------
        dataset : SFrame
            Dataset of new observations. Must include columns with the same
            names as the features used for model training, but does not require
            a target column. Additional columns are ignored.

        output_type : {'probability', 'margin', 'class'}, optional
            Form of the predictions.

        Returns
        -------
        out : SArray
            An SArray with model predictions.
        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.predict')

        if output_type not in ['margin', 'probability', 'class']:
            raise ValueError("Output type '{}' is not supported.".format(output_type) + \
                             " Please select 'probability', 'margin', or 'class'.")

        if type(dataset) == _DataFrame:
            dataset = _SFrame(dataset)

        # special handling of vw model
        if self.get('solver') == 'vw':
            m = _vw.VWModel(self.__proxy__)
            target_column = self.get('target')
            dataset_copy = dataset[dataset.column_names()]  # SFrame copy by reference

            if target_column in dataset_copy.column_names():
                dataset_copy.remove_column(target_column)

            predictions = m.predict(dataset_copy)

            # VW predictions are raw "margins" (i.e., w^T * x + b)
            if output_type == "margin":
                return predictions

            else:
                # We need to transform it back into the [0,1] range through
                # the logistic link function.
                predictions = predictions.apply(lambda x : 1 - 1/(1+math.exp(x)))

                if output_type == "probability":
                    return predictions
                else:
                    predictions = predictions.apply(lambda x: x >= 0.5)
                    return predictions


        # Get metadata from C++ object
        opts = {'model': self.__proxy__,
                'model_name': self.__name__,
                'test_data': dataset,
                'output_type': output_type}

        ## Compute the predictions
        init_opts = _graphlab.toolkits.main.run('regression_predict_init', opts)
        opts.update(init_opts)
        target = _graphlab.toolkits.main.run('regression_predict', opts)
        return _SArray(None, _proxy=target['predicted'])

    def evaluate(self, dataset):
        """
        Evaluate the model by making predictions of target values and comparing
        these to actual values.

        Two metrics are used to evaluate logistic regression. The confusion
        table contains the cross-tabulation of actual and predicted classes for
        the target variable. Classification accuracy is the fraction of examples
        whose predicted and actual classes match.

        Parameters
        ----------
        dataset : SFrame
            Dataset of new observations. Must include columns with the same
            names as the target and features used for model training. Additional
            columns are ignored.

        Returns
        -------
        out : dict
            Dictionary of evaluation results. The dictionary keys are *accuracy*
            and *confusion_table*.

        References
        ----------
        - `Wikipedia - confusion matrix
          <http://en.wikipedia.org/wiki/Confusion_matrix>`_
        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.evaluate')

        target_column = self.get('target')
        if target_column not in dataset.column_names():
            raise ToolkitError, \
                "Input dataset must contain a target column for " \
                "evaluation of prediction quality."

        target_type = dataset[target_column].dtype()
        if not target_type == int:
            raise TypeError("Target column must be type int")

        if self.get('solver') == 'vw':
            m = _vw.VWModel(self.__proxy__)
            dataset_copy = dataset[dataset.column_names()]  # SFrame copy by reference

            # If the dataset contains a column with the same heading as the
            # target column seen during training, then VW dictates that
            # this column must contain only +1/-1's.  Otherwise VW throws
            # an error.  To be consistent with the rest of the logistic
            # regression API, we'll handle the case where the target column
            # contains 0s or 1s. If not, we leave it as is, and allow VW to
            # fail.
            if set(dataset_copy[target_column]) == set([0, 1]):
                dataset_copy[target_column] = dataset_copy[target_column] * 2 - 1

            evaluation_results = m.evaluate(dataset_copy)
            return evaluation_results

        targets = dataset[target_column]
        predictions = self.predict(dataset)
        accuracy = _graphlab.evaluation.accuracy(targets, predictions)
        confusion_matrix = _graphlab.evaluation.confusion_matrix(targets, predictions)
        return {'accuracy': accuracy,
                'confusion_table': confusion_matrix}

    def training_stats(self):
        """
        Return a dictionary containing statistics collected during model
        training. These statistics are also available with the ``get`` method,
        and are described in more detail in the documentation for that method.
        """
        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.training_stats')

        solver = self.get('solver')
        # Hard code the options exposed from the VW model proxy.
        if solver == 'vw':
            training_stats_fields = ['train_iters', 'train_time']
            return dict(zip(training_stats_fields,
                        map(self.get, training_stats_fields)))
        else:
            return super(LogisticRegressionModel, self).training_stats()

    def list_fields(self):
        """
        List the fields stored in the model, including data, model, and training
        options. Each field can be queried with the ``get`` method. Note: the
        list of queryable fields is different when the solver is ``vw``.

        Returns
        -------
        out : list
            List of fields queryable with the ``get`` method.
        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.list_fields')

        solver = self.get('solver')

        if solver == 'vw':
            return ['train_iters',
                    'l1_penalty',
                    'l2_penalty',
                    'max_iterations',
                    'solver',
                    'step_size',
                    'target',
                    'train_time']
        else:
            opts = {'model': self.__proxy__,
                    'model_name': self.__name__}
            response = _graphlab.toolkits.main.run('regression_list_keys', opts)
            return sorted(response.keys())
