##\internal
"""@package graphlab.toolkits

This module defines the (internal) utility functions used by the toolkits.
"""

import threading, json, urllib, sys, logging

# try import ipython
# code adapted from
# http://stackoverflow.com/questions/15341757/how-to-check-that-pylab-backend-of-matplotlib-runs-inline/17826459#17826459
has_ipython = False
has_ipython_pylab_inline = False
has_pyplot = False
try:
    import IPython
    import IPython.display
    import IPython.core.display
    import pylab as plt

    cfg = IPython.get_ipython().config

    # Caution: cfg is an IPython.config.loader.Config
    if cfg['IPKernelApp']:
        has_ipython = True
        if cfg['IPKernelApp']['pylab'] == 'inline':
            has_ipython_pylab_inline = True
except:
    pass

try:
    import matplotlib.pyplot as pyplt
    if pyplt.get_backend().lower() != "tkagg":
        has_pyplot = True
    else:
        sys.stderr.write('Incremental plotting is not supported when TKAgg is used as the backend\n')
except:
    pass


class WorkerThread(threading.Thread):
    """
    Thread object which repeats a given task periodically until
    join is called.
    """
    def __init__(self, work, finish, interval=1):
        super(WorkerThread, self).__init__()
        self._work = work
        self._finish = finish
        self._interval = interval
        self.cv = threading.Condition()
        self.stoprequest = threading.Event()
        self.current_val = None

    def run(self):
        self.cv.acquire()
        # wait one interval round first. This means
        # if the toolkit returns within one interval,
        # nothing is ever plotted.
        self.cv.wait(self._interval)
        display_round = 0
        while not self.stoprequest.isSet():
            self.current_val = self._work(display_round, self.current_val)
            display_round = display_round + 1
            sys.stdout.flush()
            self.cv.wait(self._interval)
        self.cv.release()

    def join(self, timeout=None):
        self.cv.acquire()
        self.stoprequest.set()
        self.cv.notify()
        self.cv.release()
        super(WorkerThread, self).join(timeout)
        self._work(-1, self.current_val)
        self._finish()


# The worker function used for displaying only text output.
# This is the ultimate fallback for all the other plotting methods.
def text_display_function(display_round_unused, url, old_value=None):
    try:
        data_str = urllib.urlopen(url).read()
        data_arr = json.loads(data_str)
        if (not old_value is None and len(data_arr) and data_arr == old_value):
            print "."
            sys.stdout.flush()
        else:
            for record in data_arr:
                i = len(record['record']) - 1
                name = record['name']
                xlab = record['xlab']
                ylab = record['ylab']
                xy = record['record'][i]
                print "[%s] %s: %s\t%s: %s" % (name, xlab, xy[0], ylab, xy[1])
                sys.stdout.flush()
        return data_arr
    except ValueError:
        print 'Cannot decode metrics into json object: %s' % data_str
    except:
        print sys.exc_info()[0]


def create_text_progress_thread(url, interval=1):
    """
    Returns a WorkerThread which print the latest
    entry in an json array obtained from an url.

    Raise
    -----
    ValueError: If the worker cannot parse the content into json object.
    Error: Other errors like connection drop.

    Returns
    -------
    out : WorkerThread
    """
    return WorkerThread(lambda x, y: text_display_function(x, url, y), finish=lambda: None, interval=interval)


def create_ipython_plotting_thread(url, interval=1):
    """
    Returns a WorkerThread which creates plots from an json array obtained from an url.
    Each entry in the json array has the format of::

        {
            record: [(x1, y1), (x2, y2), ...] The actual datapoint for the plot.
            name: string The plot title.
            xlab: string The label of the x axis.
            ylab: string The label of the y axis.
        }

    Raise
    -----
    ValueError
        If the worker cannot parse the content into json object.
    Error
        Other errors like connection drop.

    Returns
    -------
    out : WorkerThread
    """
    # we need the _do_plot function to remember the figure and axis (f, axs)
    # So we need to keep a variable outside of the function.
    # But python 2.7 does not let us modify a "non-local" variable
    # So we by pass that by using a dictionary to hold the f and axs variables
    # The goal is to not create the figure until we actually enter the main
    # loop
    f_axs = {'f': None, 'axs': None, 'plot_create_success': True, 'nplots': 0}

    def _do_plot(display_round, old_value=None):
        # was unable to create the plot.
        # fallback to text
        if f_axs['plot_create_success'] is False:
            return text_display_function(display_round, url, old_value)
        # try to plot
        try:
            data_str = urllib.urlopen(url).read()
            data_arr = json.loads(data_str)
            n = len(data_arr)
            # do not plot until we have at least 2 points
            longestplot = 0
            for i in xrange(0, n):
                longestplot = max(longestplot, len(data_arr[i]['record']))
            if ((not old_value is None and n > 0 and data_arr == old_value) or longestplot < 2):
                print "."
                sys.stdout.flush()
            else:
                if (f_axs['nplots'] != n):
                    try:
                        f, axs = plt.subplots(1, n)
                        f.set_size_inches(8 * n, 5)
                        f_axs['f'] = f
                        f_axs['axs'] = axs
                        f_axs['nplots'] = n
                        sys.stdout.flush()
                    except:
                        f_axs['plot_create_success'] = False
                        return

                f = f_axs['f']
                axs = f_axs['axs']
                IPython.core.display.clear_output(True)
                for i in xrange(0, n):
                    xy = data_arr[i]['record']
                    name = data_arr[i]['name']
                    xlab = data_arr[i]['xlab']
                    ylab = data_arr[i]['ylab']
                    if n == 1:
                        ax = axs
                    else:
                        ax = axs.flat[i]
                    ax.cla()
                    ax.plot([pair[0] for pair in xy], [pair[1] for pair in xy])
                    ax.set_title(name)
                    ax.set_xlabel(xlab)
                    ax.set_ylabel(ylab)
                IPython.core.display.display(f)
            return data_arr
        except ValueError:
            print 'Cannot decode metrics into json object: %s' % data_str
        except:
            print sys.exc_info()[0]

    def _do_finish():
        IPython.core.display.clear_output(True)

    return WorkerThread(_do_plot, finish=_do_finish, interval=interval)


def create_matplotlib_plotting_thread(url, interval=1):
    """
    Returns a WorkerThread which creates plots from an json array obtained from an url.
    Each entry in the json array has the format of::

        {
            record: [(x1, y1), (x2, y2), ...] The actual datapoint for the plot.
            name: string The plot title.
            xlab: string The label of the x axis.
            ylab: string The label of the y axis.
        }

    Raises
    ------
    ValueError
        If the worker cannot parse the content into json object.
    Error
        Other errors like connection drop.

    Returns
    -------
    out : WorkerThread
    """

    # we need the _do_plot function to remember the figure and axis (f, axs)
    # So we need to keep a variable outside of the function.
    # But python 2.7 does not let us modify a "non-local" variable
    # So we by pass that by using a dictionary to hold the f and axs variables
    # The goal is to not create the figure until we actually enter the main
    # loop
    f_axs = {'f': None, 'axs': None, 'plot_create_success': True, 'nplots': 0}

    def _do_plot(display_round, old_value=None):
        # was unable to create the plot.
        # fallback to text
        if f_axs['plot_create_success'] is False:
            return text_display_function(display_round, url, old_value)

        # try to plot
        try:
            data_str = urllib.urlopen(url).read()
            data_arr = json.loads(data_str)
            n = len(data_arr)
            # do not plot until we have at least 2 points
            longestplot = 0
            for i in xrange(0, n):
                longestplot = max(longestplot, len(data_arr[i]['record']))
            if ((not old_value is None and n > 0 and data_arr == old_value) or longestplot < 2):
                print "."
                sys.stdout.flush()
            else:
                if (f_axs['nplots'] != n):
                    try:
                        f, axs = pyplt.subplots(nrows=1, ncols=n)
                        f.set_size_inches(8 * n, 5)
                        f.show()
                        f_axs['f'] = f
                        f_axs['axs'] = axs
                        f_axs['nplots'] = n
                    except:
                        f_axs['plot_create_success'] = False
                        return

                f = f_axs['f']
                axs = f_axs['axs']
                for i in xrange(0, n):
                    xy = data_arr[i]['record']
                    name = data_arr[i]['name']
                    xlab = data_arr[i]['xlab']
                    ylab = data_arr[i]['ylab']
                    if n == 1:
                        ax = axs
                    else:
                        ax = axs.flat[i]
                    ax.cla()
                    ax.plot([pair[0] for pair in xy], [pair[1] for pair in xy])
                    ax.set_title(name)
                    ax.set_xlabel(xlab)
                    ax.set_ylabel(ylab)
                if f:
                    f.canvas.draw()
        except ValueError:
            print 'Cannot decode metrics into json object: %s' % data_str
        except:
            print sys.exc_info()

    return WorkerThread(_do_plot, finish=lambda: None, interval=interval)


# automatically detects if there is ipython or not, and set up plotting appropriately
def create_auto_plotting_thread(url, interval):
    if has_ipython_pylab_inline is True:
        return create_ipython_plotting_thread(url=url, interval=interval)
    elif has_pyplot is True:
        return create_matplotlib_plotting_thread(url=url, interval=interval)
    else:
        logging.getLogger(__name__).warning("Progress plotting not supported.")
        return None
        # return create_text_progress_thread(url=url, interval=interval)
