
from graphlab.data_structures.sgraph import SGraph
import graphlab.toolkits.main as main
from graphlab.toolkits.model import Model


class PagerankModel(Model):
    """
    A model object containing the pagerank value for each vertex.

    An instance of this model can be created using :func:`graphlab.pagerank.create`.
    Do NOT construct the model directly.
    """
    def __init__(self, input_params, model):
        '''__init__(self)'''
        self.__proxy__ = model
        self._input = input_params
        self._fields = ['pagerank', 'runtime', 'total_pagerank', 'graph']

    def list_fields(self):
        """
        List of fields stored in the model. Each of these fields can be queried
        using the ``get`` function.

        Returns
        -------
        out : list
            A list of fields that can be queried using the ``get`` method.
        """

        return self._fields

    def get(self, field):
        """
        Get the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained programmatically using the
        :func:`~graphlab.graph_analytics.ConnectedComponentModel.list_fields`
        method.

        Each of these fields can be queried in one of two ways:

        >>> out = m['field']
        >>> out = m.get('field')  # equivalent to previous line

        +-----------------+----------------------------------------------------+
        |      Field      | Description                                        |
        +=================+====================================================+
        | pagerank        | SFrame with the pagerank of each vertex            |
        +-----------------+----------------------------------------------------+
        | graph           | Input graph with pagerank as a vertex property     |
        +-----------------+----------------------------------------------------+

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out : [various]
            The current value of the requested field.
       """

        if (field == 'pagerank'):
            return SGraph(_proxy=self.__proxy__.get('__graph__')).select_fields(['pagerank']).get_vertices()
        elif(field == 'graph'):
            return SGraph(_proxy=self.__proxy__.get('__graph__'))
        else:
            return self.__proxy__.get(field)

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""

        ret = {'hyperparams': self._input}
        for key in ['runtime', 'total_pagerank', 'delta']:
            ret[key] = self.get(key)
        return ret

    def _get_wrapper(self):
        return lambda m: PagerankModel(self._input, m)


def create(graph, reset_prob=0.15, threshold=0.01, max_iterations=20, verbose=True, plot=False):
    """
    Compute the PageRank for each vertex in the graph. Return a model object
    with total PageRank as well as the PageRank value for each vertex in the
    graph.

    Parameters
    ----------
    graph : SGraph
        The graph on which to compute the pagerank value.

    reset_prob : float, optional
        Probability that a random surfer jumps to an arbitrary page.

    threshold : float, optional
        Threshold for convergence, measured in the L1 norm
        (the sum of absolute value) of the delta of each vertex's
        pagerank value.

    max_iterations: int, optional
        The maximun number of iterations to run.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : Model
        A Model object that contains the total and individual PageRank.

    References
    ----------
    - `Wikipedia - PageRank <http://en.wikipedia.org/wiki/PageRank>`_
    - Page, L., et al. (1998) `The PageRank Citation Ranking: Bringing Order to
      the Web <http://ilpubs.stanford.edu:8090/422/1/1999-66.pdf>`_.

    Examples
    --------
    If given an :class:`~graphlab.SGraph` ``g``, we can create
    a :class:`~graphlab.pagerank.PageRankModel` as follows:

    >>> pr = pagerank.create(g)

    We can obtain the page rank corresponding to each vertex in the graph ``g``
    using:

    >>> pr_out = p['pagerank']     # SFrame
    """
    if not isinstance(graph, SGraph):
        raise TypeError('graph input must be a SGraph object.')

    if plot is True:
        print "The plot functionality for pagerank is not yet implemented."

    opts = {'threshold': threshold, 'random_jump_prob': reset_prob,
            'max_iter': max_iterations, 'graph': graph.__proxy__}
    params = main.run('pagerank', opts, verbose, plot)
    return PagerankModel({'threshold': threshold, 'reset_prob': reset_prob},
                         params['model'])
