
from graphlab.data_structures.sgraph import SGraph
import graphlab.toolkits.main as main
from graphlab.toolkits.model import Model


class TriangleCountingModel(Model):
    """
    A model object containing the traingle_count for each vertex, and the total
    number of triangles.

    An instance of this model can be created using :func:`graphlab.triangle_counting.create`.
    Do NOT construct the model directly.
    """
    def __init__(self, input_params, model):
        '''__init__(self)'''
        self.__proxy__ = model
        self._input = input_params
        self._fields = ['num_triangles', 'runtime', 'triangle_count', 'graph']

    def list_fields(self):
        """
        List of fields stored in the model. Each of these fields can be queried
        using the ``get`` function.

        Returns
        -------
        out : list
            A list of fields that can be queried using the ``get`` method.
        """

        return self._fields

    def get(self, field):
        """
        Get the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained programmatically using the
        :func:`~graphlab.graph_analytics.TriangleCountingModel.list_fields`

        method.

        Each of these fields can be queried in one of two ways:

        >>> out = m['field']
        >>> out = m.get('field')  # equivalent to previous line

        +----------------+-----------------------------------------------------+
        |      Field     | Description                                         |
        +================+=====================================================+
        | triangle_count | SFrame with the triangle count of each vertex       |
        +----------------+-----------------------------------------------------+
        | graph          | Input graph with the count as a vertex property     |
        +----------------+-----------------------------------------------------+

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out : [various]
            The current value of the requested field.

       """

        if (field == 'triangle_count'):
            return SGraph(_proxy=self.__proxy__.get('__graph__')).select_fields(['triangle_count']).get_vertices()
        elif (field == 'graph'):
            return SGraph(_proxy=self.__proxy__.get('__graph__'))
        else:
            return self.__proxy__.get(field)

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""

        ret = {'hyperparams': self._input}
        for key in ['runtime', 'num_triangles']:
            ret[key] = self.get(key)
        return ret

    def _get_wrapper(self):
        return lambda m: TriangleCountingModel(self._input, m)


def create(graph, verbose=True, plot=False):
    """
    Compute the number of triangles each vertex belongs to, ignoring edge
    directions. A triangle is a complete subgraph with only three vertices.
    Return a model object with total number of triangles as well as the triangle
    counts for each vertex in the graph.

    Parameters
    ----------
    graph : SGraph
        The graph on which to compute triangle counts.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : Model
        A Model object that contains the total and individual triangle counts.

    Examples
    --------
    If given an :class:`~graphlab.SGraph` ``g``, we can create
    a :class:`~graphlab.traingle_counting.TriangleCountingModel` as follows:

    >>> tc = triangle_counting.create(g)

    We can obtain the number of triangles that each vertex in the graph ``g``
    is present in:

    >>> tc_out = tc['triangle_count']  # SFrame

    """
    if not isinstance(graph, SGraph):
        raise TypeError('graph input must be a SGraph object.')

    if plot is True:
        print "The plot functionality for triangle counting is not yet implemented."

    params = main.run('triangle_counting', {'graph': graph.__proxy__}, verbose, plot)
    return TriangleCountingModel({}, params['model'])
