"""
This package provides the GraphLab recommender system toolkit.

The toolkit includes a module with several recommender system models that can be
trained and used to make new recommendations. For detail beyond this overview,
please see the tutorials at http://www.graphlab.com/learn.

The GraphLab recommender models accept an
:class:`~graphlab.SFrame` or a Pandas DataFrame. Our
example dataset is stored in a csv file, which is retrieved from an Amazon S3
address. The most basic type of dataset has just two columns: user ID and item
ID. The fastest way to use the GraphLab recommender system via the
:py:func:`~graphlab.recommender.create` function:

.. sourcecode:: python

    >>> from graphlab import recommender, SFrame
    >>> data = SFrame('s3:///graphlab-sample-data/example.csv')
    >>> m = recommender.create(data, user_column='user_id', item_column='item_id')
    >>> recs = m.recommend(users=['u1', 'u3', 'u15'])
    >>> print recs

When no ``target_column`` is provided, then by default, this returns a
:py:class:`~graphlab.recommender.ItemSimilarityModel`, which computes the similarity
between each pair of items and recommends items to each user that
are closest to items she has already used or liked. It measures item similarity
with either Jaccard or cosine distance, which can be set manually using a
keyword argument called ``similarity_type``:

.. sourcecode:: python

    >>> m = recommender.create(data, user_column='user_id', item_column='item_id',
                               method='item_similarity', similarity_type='jaccard')

When a ``target_column`` is provided, the default GraphLab recommender is a
matrix factorization model. The matrix factorization model can also be
called explicitly, with much greater flexibility, by setting
``method=matrix_factorization``. Other arguments can be provided, such
as ``n_factors`` or ``regularization`` when using a
:py:class:`~graphlab.recommender.MatrixFactorizationModel`.

.. sourcecode:: python

    >>> m = recommender.create(data,
                               user_column='user_id',
                               item_column='item_id',
                               target_column='rating',
                               method='matrix_factorization',
                               n_factors=7, regularization=0.05, nmf=True)

All model objects in the :mod:`~graphlab.recommender` module expose a common set of
methods, such as :py:func:`~graphlab.recommender.MatrixFactorizationModel.recommend`
and :py:func:`~graphlab.recommender.MatrixFactorizationModel.evaluate`.

See http://www.graphlab.com/products/create/tutorials.html for tutorials on
building and using recommenders.

"""

import popularity
import matrix_factorization
import linear_regression
import item_similarity
import item_means

from item_similarity import ItemSimilarityModel
from matrix_factorization import MatrixFactorizationModel
from popularity import PopularityModel
from item_means import ItemMeansModel
from linear_regression import LinearRegressionModel
from factorization_model import FactorizationModel

from graphlab.toolkits.recommender.util import create
from graphlab.toolkits.recommender.util import compare_models
from graphlab.toolkits.recommender.util import precision_recall_by_user
from graphlab.toolkits.recommender.util import random_split_by_user



