"""
Methods for creating models that rank items according to their similarity
to other items.
"""
import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits.recommender.recommender import RecommenderModel
from graphlab.data_structures.sframe import SFrame as _SFrame
from pandas import DataFrame as _DataFrame
import warnings
import logging

def create(dataset,
           user_column='user_id', item_column='item_id', target_column=None,
           user_data=None, item_data=None,
           similarity_type='jaccard',
           only_top_k=0, threshold=0.0,
           threshold_degree=None, threshold_window=None,
           verbose=True, plot=False, **method_options):
    """
    Create an :class:`~graphlab.recommender.item_similarity.ItemSimilarityModel`.
    This scores an item according to its similarity to other items observed for the
    user in question. The similarity can be computed based on the observed users
    for each item, e.g. the rating given to both items by users who have rated
    both items.

    NOTE: This will be deprecated soon. Please use
    graphlab.recommender.create instead.


    Parameters
    ----------
    dataset : SFrame
        The dataset to use for training the model.

    user_column : string
        The column name of the dataset that corresponds to user id.

    item_column : string
        The column name of the dataset that corresponds to item id.

    target_column : string
        If a score is present in the information -- such as a rating -- then
        this specifies the name of the column in the data to use as the target variable.

    user_data : SFrame
        Side information for the users.  This SFrame must have a column named
        with the  user column name given above. It provides any amount of
        additional user-specific information.

    item_data: SFrame
        Side information for the items. This SFrame must have a column named
        with the item column name given above. It provides any amount of
        additional item-specific information.

    similarity_type : {'jaccard', 'cosine'}, optional
        Measure of similarity between two items. The allowed functions are:
        - 'jaccard' (default)
        - 'cosine'

    only_top_k : int, optional
        If this value is greater than 0, only the distances to each item's
        k most similar items are stored and used for predictions.
        When there are relatively few items given the number of observations,
        it can be faster to precompute and store all item-to-item similarity
        scores. When many unique items exist, using k>0 can help reduce
        the amount of memory required.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    threshold : float, optional
        Any items that have similarity below this threshold are ignored during
        prediction.

    threshold_degree : int, optional
        Skip all items with fewer than ``threshold_degree`` observations
        during prediction. This option is only available of ``only_top_k`` > 0.

    threshold_window : float, optional
        Experimental: When ``only_top_k`` > 0, this allows one to skip
        computing similarities for item pairs (i, j) whenever
        max(d_i, d_j) / min(d_i, d_j) > ``threshold_window``, where
        d_i is the number of observations for item i.

    -------
    out : ItemSimilarityModel
        A trained ItemSimilarityModel.

    Examples
    --------
    If given an :class:`~graphlab.SFrame` ``sf`` with columns ``user_id`` and
    ``item_id``, then we can create a
    :class:`~graphlab.item_similarity.ItemSimilarityModel` as follows:

    >>> from graphlab import recommender
    >>> m = recommender.item_similarity.create(sf, 'user_id', 'item_id')

    With this model object one can make recommendations for the unique users in ``sf``:

    >>> recs = m.recommend(sf)

    The model can be saved to disk as follows:

    >>> m.save(filename)

    For more, see the documentation for
    :class:`~graphlab.recommender.item_similarity.ItemSimilarityModel`.

    References
    ----------
    - Sarwar, B., et al. (2001) `Item-Based Collaborative Filtering
      Recommendation Algorithms
      <http://files.grouplens.org/papers/www10_sarwar.pdf>`_. Proceedings of the
      10th International Conference on World Wide Web. pp. 285-295.
    - `Wikipedia - Jaccard index <http://en.wikipedia.org/wiki/Jaccard_index>`_
    - `Wikipedia - cosine similarity <http://en.wikipedia.org/wiki/Cosine_similarity>`_
    """

    _mt._get_metric_tracker().track('toolkit.recsys.item_similarity.create')

    logging.warning("item_similarity.create will be deprecated soon. Please switch to recommender.create")

    if not isinstance(dataset, (_DataFrame, _SFrame)):
        raise TypeError('dataset input must be a pandas.DataFrame or SFrame')

    if type(dataset) != _SFrame:
        dataset = _SFrame(dataset)

    if plot is True:
        print "The plot functionality for item similarity is not yet implemented."
        plot = False

    method_options = {'similarity_type': similarity_type,
                      'only_top_k': only_top_k,
                      'threshold': threshold}

    # Add threshold options if provided by the user.
    if threshold_degree is not None:
        method_options.update({'threshold_lower_degree': threshold_degree})

    if threshold_window is not None:
        method_options.update({'threshold_window_degree': threshold_window})


    m = _graphlab.recommender.create(dataset,
            user_column=user_column, item_column=item_column, target_column=target_column,
            user_data=user_data, item_data=item_data,
            verbose=verbose,
            method='item_similarity',
            **method_options)

    return m


class ItemSimilarityModel(RecommenderModel):
    """
    A model that ranks an item according to its similarity to other items
    observed for the user in question. The similarity can be computed based on
    the observed users for each item, e.g. the rating given to both items by
    users who have rated both items.

    An instance of this model can be created using
    :func:`create(..., method='item_similarity') <graphlab.recommender.create>`.
    Do NOT construct the model directly.

    NOTE: This will be deprecated soon. Please use
    :func:`create(..., method='item_similarity') <graphlab.recommender.create>`
    instead.
    """

    def __init__(self, model_proxy):
        '''__init__(self)'''
        self.__proxy__ = model_proxy

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return ItemSimilarityModel(model_proxy)
        return model_wrapper

