"""
Methods for performing linear regression.  See
graphlab.linear_regression.create for additional documentation.
"""

import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits.recommender.recommender import RecommenderModel
from graphlab.data_structures.sframe import SFrame as _SFrame
from pandas import DataFrame as _DataFrame
import logging

def create(dataset,
           user_column='user_id', item_column='item_id',
           user_data=None, item_data=None,
           reg=1e-5, sgd_step_size=0, max_iterations=50,
           verbose=True, plot=False,
           random_seed = 0,
           **kwargs):
    """
    Create a
    :class:`~graphlab.recommender.linear_regression.LinearRegressionModel`. This
    can be used to score (user, item) pairs and make recommendations.

    Linear regression creates a linear function that predicts a continuous
    target value given the user, item, and any additional features.  Unlike
    matrix factorization, this model can use any number of additional features
    such as user or item attributes to help make an accurate prediction.

    The parallel optimization method used to build this model is inherently
    random, so different calls to `create()` may return slightly different
    models.

    In addition to the `user` and `item` features,
    :class:`~.LinearRegressionModel` also uses any additional columns in
    `dataset` to build the model and make predictions.  These features can
    include additional attributes of the user or the item in the observation.
    Additionally, observation-specific information such as the time of day the
    user rated the item can also be included. Values for these side features
    must be present both when the model is created and when predictions are
    made.

    The exact function of these features in the regression model is determined
    by the type of the :class:`~graphlab.SFrame` column holding the feature
    values.  If these features are strings or integer types, they are treated as
    categorical variables alongside users and items; if they are typed as
    floating point numbers, then they are treated as continuous values.

    NOTE: This will be deprecated soon. Please use
    graphlab.recommender.create instead.


    Parameters
    ----------
    dataset : SFrame/DataFrame
        The dataset to use for training the model.

    user_column : string
        The column name of the dataset that corresponds to user id.

    item_column : string
        The column name of the dataset that corresponds to item id.

    target_column : string
        The model will be trained to predict the values in this column.

    reg : float, optional
        Regularization weight.

    sgd_step_size : float, optional
        Step size for stochastic gradient descent. Smaller values may
        add accuracy but cause the model to converge more slowly.  If
        zero (default), the step size is chosen by trying several options
        on a small subset of the data.

    max_iterations : int, optional
        Maximum number of iterations through the data.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    random_seed : integer, optional
        The random seed used to choose a random starting point with
        which to initialize the model.

    Returns
    -------
    out : LinearRegressionModel
        A trained model.

    Examples
    --------
    If given an :class:`~graphlab.SFrame` ``sf`` with columns ``user_id`` and ``item_id`` and
    a column we would like to predict called ``target``, then we can
    create a :class:`~graphlab.linear_regression.LinearRegressionModel` as follows:

    >>> from graphlab import recommender
    >>> m = recommender.linear_regression.create(sf, 'user_id', 'item_id', target='target')

    With this model object one can make recommendations for the unique users in ``sf``:

    >>> recs = m.recommend(sf)

    The model can be saved to disk as follows:

    >>> m.save(filename)

    For more, see the documentation for
    :class:`~graphlab.recommender.linear_regression.LinearRegressionModel`.

    References
    ----------
    - `Wikipedia - linear regression (general)
      <http://en.wikipedia.org/wiki/Linear_regression>`_
    """

    _mt._get_metric_tracker().track('toolkit.recsys.linear_regression.create')
    logging.warning("This method will be deprecated soon. Please switch to recommender.create")

    if not isinstance(dataset, (_DataFrame, _SFrame)):
        raise TypeError('dataset input must be a pandas.DataFrame or SFrame')

    if type(dataset) != _SFrame:
        dataset = _SFrame(dataset)

    if plot is True:
        print "The plot functionality for linear_regression is not yet implemented."
        plot = False

    method_options = {'regularization': reg,
                      'sgd_step_size': sgd_step_size,
                      'max_iterations':max_iterations}
    method_options.update(kwargs)

    m = _graphlab.recommender.create(dataset,
            user_column=user_column, item_column=item_column,
            user_data=user_data, item_data=item_data,
            verbose=verbose,
            method='linear_model',
            **method_options)

    return m


class LinearRegressionModel(RecommenderModel):
    """
    A linear regression model that learns a coefficient for each user and
    a coefficient for each item, as well as a global offset term, and tries
    to predict the given user-item ratings.  The "user_id" and "item_id"
    columns in the input data are treated as categorical variables.

    See Wikipedia for a description of linear regression:
    http://en.wikipedia.org/wiki/Linear_regression

    An instance of this model can be created using
    :func:`create(..., method='linear_model') <graphlab.recommender.create>`.
    Do NOT construct the model directly.

    NOTE: This will be deprecated soon. Please use
    :func:`create(..., method='linear_model') <graphlab.recommender.create>`
    instead.
    """

    def __init__(self, model_proxy):
        '''__init__(self)'''
        self.__proxy__ = model_proxy

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return LinearRegressionModel(model_proxy)
        return model_wrapper
