/** @jsx React.DOM */

define('plot_graph',
      ['react', 'd3', 'common_util', 'common_controls'],
      function(React, d3, Util, Controls) {

  var plotSize = 600;
  var plotPadding = 50;

  // Subcomponents of graph drawing
  var Edge = React.createClass({displayName: 'Edge',
    render: function() {
      var e = this.props.edge;
      var lines = [];
      for (var i=0; i<e.points.length-1; i++) {
        var p1 = e.points[i];
        var p2 = e.points[i+1];
        var x1 = this.props.scaleX(p1[0]);
        var y1 = this.props.scaleY(p1[1]);
        var x2 = this.props.scaleX(p2[0]);
        var y2 = this.props.scaleY(p2[1]);
        var props = {
          key: i,
          x1: x1,
          y1: y1,
          x2: x2,
          y2: y2,
          style: {
            stroke: this.props.ecolor,
            strokeWidth: this.props.ewidth
          }
        };
        var line = React.DOM.line(props);
        lines.push(line);

        if (this.props.arrows && i == e.points.length-2) {
          // draw an arrow on the last segment
          // TODO use an svg <marker> when React supports it
          
          var dx = x2 - x1; // change in x over whole segment
          var dy = y2 - y1; // change in y over whole segment
          var m = dy/dx; // slope of whole segment
          var s = this.props.ewidth * 5;
          var L = Math.sqrt((dx*dx)+(dy*dy)); // length of the whole edge
          var f = (s * 2) / L; // fraction of the edge occupied by arrow

          // shift the arrows by vertexRadius px across the line segment
          // so the tip aligns with the the vertex
          // (the segment extends into the center of the vertex)
          var vertexOffset = (this.props.vertexRadius - 2) / L; // fraction of the edge to move arrow tip by
          var p_tip = [x2 - (dx * vertexOffset), y2 - (dy * vertexOffset)]; // tip of the arrow

          // calculate p_left and p_right position (the two points on the arrow
          // triangle other than the tip)
          var p_back = [p_tip[0] - (dx * f), p_tip[1] - (dy * f)]; // position on the segment intersecting the back of the arrow
          var xOffset = Math.sqrt(((s*s)*(m*m))/((m*m)+1));
          var yOffset = (-1/m)*xOffset;
          var p_left = [p_back[0] + xOffset, p_back[1] + yOffset];
          var p_right = [p_back[0] - xOffset, p_back[1] - yOffset];

          // create a polygon with those points
          var points = [p_tip, p_left, p_right];
          points = points.map(function(p) {
            // transform to an SVG points string
            return p.join(',');
          }).join(' ');
          lines.push(
            React.DOM.polygon(
              {key:"arrow",
              points:points,
              fill:this.props.ecolor,
              strokeWidth:0}
            )
          );
        }
      }

      return (
        React.DOM.g( {onMouseOver:this.props.onMouseOver, onMouseOut:this.props.onMouseOut}, 
          lines
        )
      );
    }
  });

  var Vertex = React.createClass({displayName: 'Vertex',
    render: function() {
      var x1 = this.props.scaleX(this.props.vertex.x);
      var y1 = this.props.scaleY(this.props.vertex.y);
      Util.assert(x1 >= 0);
      Util.assert(y1 >= 0);
      var fill = this.props.vcolor;
      if (this.props.vertex.value in this.props.highlight) {
        fill = this.props.highlight[this.props.vertex.value];
      }
      var props = {
        onMouseOver: this.props.onMouseOver,
        onMouseOut: this.props.onMouseOut,
        r: this.props.vertexRadius,
        cx: x1,
        cy: y1,
        style: {
          'fill': fill
        }
      };
      return React.DOM.circle(props);
    }
  });

  var Label = React.createClass({displayName: 'Label',
    render: function() {
      // draw the same text twice, first in 5px white then in 1px black.
      // this gives a nice white outline around black text.
      return (
        React.DOM.g(null, 
          React.DOM.text(
            {x:this.props.x,
            y:this.props.y,
            style:{
              textAnchor: 'middle',
              fill: 'white',
              stroke: 'white',
              strokeWidth: 5,
              opacity: 0.9
            }}
          , 
            this.props.text
          ),
          React.DOM.text(
            {x:this.props.x,
            y:this.props.y,
            style:{
              textAnchor: 'middle',
              fill: 'black'
            }}
          , 
            this.props.text
          )
        )
      );
    }
  });

  // Layout functions
  var layoutAlgorithms = {
    forceDirected: function(vertices, edges) {
      var force = d3.layout.force()
        .nodes(vertices)
        .links(edges)
        .charge(-60)
        .linkDistance(20)
        .start();
      for (var i=0; force.alpha() > 0 && i<1000; i++) {
        force.tick();
      }
      force.stop();
      force.links().map(function(e) {
        e['points'] = [
          [e.source.x, e.source.y],
          [e.target.x, e.target.y]
        ];
      });
      return {
        'vertices': force.nodes(),
        'edges': force.links()
      };
    }/*,
    hierarchical: function(vertices, edges) {
      var g = new dagre.Digraph();

      for (var i=0; i<vertices.length; i++) {
        g.addNode(vertices[i].value, { label: String(vertices[i].value), width: 1000, height: 1000 });
      }
      for (var i=0; i<edges.length; i++) {
        g.addEdge(null, vertices[edges[i].source].value, vertices[edges[i].target].value);
      }
      var layout = dagre.layout().run(g);
      var nodeElements = [],
          edgeElements = [];
      var idx = 0;
      layout.eachNode(function(u, value) {
        vertices[idx]['x'] = value.x;
        vertices[idx++]['y'] = value.y;
      });
      var edgeIdx = 0;
      layout.eachEdge(function(e, u, v, value) {
        var points = [[layout.node(u).x, layout.node(u).y]];
        for (var i=0; i<value.points.length; i++) {
          points.push([value.points[i].x, value.points[i].y]);
        }
        points.push([layout.node(v).x, layout.node(v).y]);
        edges[edgeIdx++]['points'] = points;
      });
      return {
        'vertices': vertices,
        'edges': edges
      };
    }*/
  };

  return {
    GraphLayout: React.createClass({displayName: 'GraphLayout',
      statics: {
        'layoutAlgorithms': layoutAlgorithms,
        'layout': function(props) {
          var getVertex = function(value) {
            // reverse lookup a vertex by value
            // (for edge mapping)
            var vertex = props.vertices.indexOf(value);
            Util.assert(vertex != -1);
            return vertex;
          };
          return props.layoutAlgorithm(
            props.vertices.map(function(v, idx) {
              return {
                'value': v,
                'label': props.vlabels !== null ? props.vlabels[idx] : null
              };
            }),
            props.edges.map(function(e, idx) {
              return {
                'source': getVertex(e[0]),
                'target': getVertex(e[1]),
                'label': props.elabels !== null ? props.elabels[idx] : null
              };
            })
          );
        },
      },
      getDefaultProps: function() {
        return {
          'layoutAlgorithm': this.type.layoutAlgorithms.forceDirected
        };
      },
      getInitialState: function() {
        return {
          'graph': this.type.layout(this.props),
          'vertexHover': null,
          'edgeHover': null
        };
      },
      componentWillReceiveProps: function(nextProps) {
        this.setState({graph: this.type.layout(nextProps)});
      },
      onVertexMouseOver: function(vertex) {
        this.setState({vertexHover: vertex});
      },
      onVertexMouseOut: function() {
        this.setState({vertexHover: null});
      },
      onEdgeMouseOver: function(edge) {
        this.setState({edgeHover: edge});
      },
      onEdgeMouseOut: function() {
        this.setState({edgeHover: null});
      },
      render: function() {
        var vertexRadius = this.props.node_size;
        var graph = this.state.graph;
        var vx = graph.vertices.map(function(v) {return v.x;});
        var vy = graph.vertices.map(function(v) {return v.y;});
        var effectiveWidth = d3.max(vx) - d3.min(vx);
        var effectiveHeight = d3.max(vy) - d3.min(vy);
        // maintain aspect ratio when scaling
        var aspectRatio = effectiveWidth / effectiveHeight;
        var scaleX, _scaleY;
        if (aspectRatio > 1) {
          // wider
          scaleX = d3.scale.linear()
            .domain([d3.min(vx), d3.max(vx)])
            .range([plotPadding, plotSize - plotPadding]);
          _scaleY = d3.scale.linear()
            .domain([d3.min(vy), d3.max(vy)])
            .range([plotPadding, (plotSize - plotPadding) / aspectRatio]);
        } else {
          // taller
          scaleX = d3.scale.linear()
            .domain([d3.min(vx), d3.max(vx)])
            .range([plotPadding, (plotSize - plotPadding) * aspectRatio]);
          _scaleY = d3.scale.linear()
            .domain([d3.min(vy), d3.max(vy)])
            .range([plotPadding, plotSize - plotPadding]);
        }
        // flip Y axis (SVG 0 is on top)
        var vyScaledMax = d3.max(vy.map(_scaleY));
        var scaleY = function(y) { return (plotSize - plotPadding) - (((plotSize - (2 * plotPadding)) - vyScaledMax) + _scaleY(y)); };
        //var scaleY = _scaleY;
        return (
          React.DOM.div(null, 
            React.DOM.svg( {width:plotSize, height:plotSize}, 
              graph.edges.map(function(e, idx) {
                return Edge({
                  key: idx,
                  edge: e,
                  ewidth: this.props.ewidth,
                  ecolor: this.props.ecolor,
                  scaleX: scaleX,
                  scaleY: scaleY,
                  arrows: this.props.arrows,
                  vertexRadius: vertexRadius,
                  onMouseOver: this.onEdgeMouseOver.bind(null, idx),
                  onMouseOut: this.onEdgeMouseOut
                });
              }.bind(this)),
              graph.vertices.map(function(n, idx) {
                return Vertex({
                  key: idx,
                  vertex: n,
                  vertexRadius: vertexRadius,
                  vcolor: this.props.vcolor,
                  highlight: this.props.highlight,
                  scaleX: scaleX,
                  scaleY: scaleY,
                  onMouseOver: this.onVertexMouseOver.bind(null, idx),
                  onMouseOut: this.onVertexMouseOut
                });
              }.bind(this)),
              this.props.elabels.map(function(label, idx) {
                if (this.props.elabel_hover &&
                    this.state.edgeHover !== idx) {
                  return null;
                }
                return Label({
                  key: idx,
                  x: scaleX(d3.mean(graph.edges[idx].points.map(function(p) { return p[0]; }))),
                  y: scaleY(d3.mean(graph.edges[idx].points.map(function(p) { return p[1]; }))),
                  text: typeof label === 'number' ? Util.formatNumber(label) : label,
                });
              }.bind(this)),
              this.props.vlabels.map(function(label, idx) {
                if (this.props.vlabel_hover &&
                    this.state.vertexHover !== idx) {
                  return null;
                }
                return Label({
                  key: idx,
                  x: scaleX(graph.vertices[idx].x) + (this.props.h_offset * plotSize),
                  y: scaleY(graph.vertices[idx].y) - (this.props.v_offset * plotSize),
                  v_offset: this.props.v_offset,
                  h_offset: this.props.h_offset,
                  text: typeof label === 'number' ? Util.formatNumber(label) : label,
                });
              }.bind(this))
            )
          )
        );
      }
    })
  };
});
