"""@package graphlab.connect.aws.ec2

This module makes it easy to have a GraphLab server running in EC2.
"""

import ConfigParser
import json
import logging
import os
import time
import urllib2

import boto.ec2
from boto.exception import EC2ResponseError

import graphlab.product_key
import graphlab.connect.server as glserver
import graphlab.connect.main as glconnect
from graphlab.connect.main import __catch_and_log__

from graphlab.cython.cy_ipc import PyCommClient as Client
from graphlab.cython.cy_ipc import get_public_secret_key_pair

from graphlab_util.config import DEFAULT_CONFIG as config
from graphlab.connect import _get_metric_tracker

CONFIG_SECTION = 'AWS'
DEFAULT_CIDR_RULE = '0.0.0.0/0'  # Open to the entire internet.
DEFAULT_INSTANCE_TYPE = 'm3.xlarge'
GRAPHLAB_PORT_MIN_NUM = 9000
GRAPHLAB_PORT_MAX_NUM = 9004
JSON_BLOB_PATH_FORMAT = "/gl_api/api/v1/cloud/aws/instances/%s?client_version=%s&region=%s"
GRAPHLAB_NAME = 'GraphLab'

# Swallow all boto logging, except critical issues
logging.getLogger('boto').setLevel(logging.CRITICAL)

__LOGGER__ = logging.getLogger(__name__)

class _EC2(glserver.RemoteServer):
    """
    This class is responsible for starting up an EC2 instance.
    """
    def __init__(self, instance_type, region, CIDR_rule, auth_token, product_key, security_group_name, 
                 tags, public_key, secret_key, user_data, credentials):
        self.logger = __LOGGER__
        self.region = region
        self.auth_token = auth_token
        
        _get_metric_tracker().track(('ec2.instance_type.%s' % instance_type), value=1)
        _get_metric_tracker().track(('ec2.region.%s' % region), value=1)

        try:
            conn = boto.ec2.connect_to_region(self.region, **credentials)

            # Does the security group already exist?
            security_group = None
            for sg in conn.get_all_security_groups():
                if(security_group_name == sg.name):
                    security_group = sg
                    break

        except boto.exception.EC2ResponseError as e:
            # This is the first AWS contact, so the most likely cause is environment variables 
            # aren't set.
            self.logger.debug('EC2 Response Error: %s' % e)
            raise Exception('EC2 response error. Please verify your AWS credentials. To configure your AWS credentials use graphlab.aws.set_credentials or set the AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY environment variables')
        except boto.exception.NoAuthHandlerFound as e:
            raise Exception('AWS configuration not found. Please configure your AWS credentials using graphlab.aws.set_credentials or by setting AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY environment variables')

        # If security group doesn't exist, create it.
        if security_group is None:
            security_group = conn.create_security_group(security_group_name,
                                                        'Only open ports needed for GraphLab Server')

        # Autorize the user CIDR rules for the ports we need
        try:
            security_group.authorize('tcp', GRAPHLAB_PORT_MIN_NUM, GRAPHLAB_PORT_MAX_NUM, CIDR_rule)
        except EC2ResponseError as e:
            if not "<Code>InvalidPermission.Duplicate</Code>" in str(e):
                raise

        if (config.mode != 'PROD' and 'GRAPHLAB_TEST_AMI_ID' in os.environ and 'GRAPHLAB_TEST_ENGINE_URL' in os.environ):
            # unit-test mode, don't involve webservice to retrieve AMI, instead use environment variables
            ami_id = os.environ['GRAPHLAB_TEST_AMI_ID']
            engine_url = os.environ['GRAPHLAB_TEST_ENGINE_URL']
            self.logger.info("UNIT mode, using AMI: '%s' and engine url: '%s' when launching EC2 instance." % (ami_id, engine_url))
            json_blob = json.loads('{}')
            json_blob['ami_id'] = ami_id
            json_blob['engine_url'] = engine_url
            json_blob['os_url'] = ''
        else:
            # Get the info to start a EC2 from the GraphLab Server
            json_blob_path = JSON_BLOB_PATH_FORMAT % (instance_type, config.version, self.region)
            json_blob_url = config.graphlab_server + json_blob_path

            try:
                # set specific timeout for this web service request, lots of time spent in SSL negotiation
                graphlab_server_response = urllib2.urlopen(json_blob_url, timeout=10)
                json_blob = json.loads(graphlab_server_response.read())
            except:
                raise Exception("Unable to successfully retrieve correct EC2 image to launch for this version. This could be a temporary problem. Please try again in a few minutes. If the problem persists please contact support@graphlab.com")

        self.logger.debug("web service return: %s" % json_blob)

        if json_blob['ami_id'] is None:
            raise Exception("Unable to successfully retrieve correct EC2 image to launch. Please try again later or contact support@graphlab.com. Error received:'%s'" % json_blob['message'])
        ami_id = json_blob['ami_id']

        if self.auth_token is not None:
            json_blob['auth_token'] = self.auth_token
        else:
            json_blob['auth_token'] = ''

        if secret_key != '':
            json_blob['secret_key'] = secret_key

        # set product key in json_blob
        if product_key is None:
            product_key = ''
        json_blob['product_key'] = product_key

        user_data.update(json_blob)

        run_instances_args =  {
            'security_group_ids' : [ security_group_name ],
            'user_data' : json.dumps(user_data),
            'instance_type' : instance_type
        }

        if (config.mode != 'PROD' and 'GRAPHLAB_TEST_EC2_KEYPAIR' in os.environ):
            keypair = os.environ['GRAPHLAB_TEST_EC2_KEYPAIR']
            self.logger.info("UNIT mode, using keypair: '%s' when launching EC2 instance" % (keypair))
            run_instances_args['key_name'] = keypair

        # Launch the EC instance
        response = conn.run_instances(ami_id, **run_instances_args)

        if(not(len(response.instances) == 1)):
            raise Exception('Could not successfully start EC2 instance.')

        self.instance = response.instances[0]

        # If auth_token is not set by the input argument, the EC2 instance
        # will use its instance id as the default auth_token, so we reset
        # self.auth_token after the instance is running.
        if not self.auth_token:
            self.auth_token = self.instance.id

        self.logger.info("Launching an %s instance in the %s availability zone, with id: %s. You will be responsible for the cost of this instance." % (self.instance.instance_type, self.instance.placement, self.instance.id))

        # Wait for the host to start up
        while not self.instance.update() == 'pending':
            time.sleep(1)
        while self.instance.update() == 'pending':
            time.sleep(1)

        if self.instance.state == 'running':
            self.instance_id = self.instance.id
            server_addr = 'tcp://' + self.instance.ip_address + ':' + str(GRAPHLAB_PORT_MIN_NUM)
            self.logger.debug("EC2 instance is now running, attempting to connect to engine.")
            super(self.__class__, self).__init__(server_addr, auth_token=self.auth_token, 
                                                 public_key=public_key)
        else:   # Error case
            self.logger.error('Could not startup EC2 instance.')
            if(hasattr(self.instance, 'id')):
                self.stop()
            return 
        
        self.public_dns_name = self.instance.public_dns_name

        # Add tags to this instance.
        if(tags is None):
            tags = {}
        tags[GRAPHLAB_NAME] = ''
        conn.create_tags(self.instance.id, tags)


    def get_auth_token(self):
        return self.auth_token

    def stop(self):
        _get_metric_tracker().flush()
        self.logger.info('Terminating EC2 instance, id: %s.' % (self.instance_id))
        try:
            conn = boto.ec2.connect_to_region(self.region)
            response = conn.terminate_instances(instance_ids=[self.instance_id])
            if(not(len(response) == 1)):
                self.logger.error('Invalid response from EC2. Unable to terminate host. Please terminate '
                                'using the AWS EC2 Console.')
                return
            if response[0].update() == 'shutting-down':
                while response[0].update() == 'shutting-down':
                    time.sleep(1)
            else:
                self.logger.error('Invalid response from EC2. Unable to terminate host. Please terminate '
                                'using the AWS EC2 Console.')
        except:
            self.logger.error('Invalid response from EC2. Unable to terminate host. Please terminate '
                              'using the AWS EC2 Console.')


def _get_region_from_config(config_path = (os.path.join(os.path.expanduser("~"), ".graphlab", "config"))):
    result = None
    if (os.path.isfile(config_path)):
        config = ConfigParser.ConfigParser()
        config.read(config_path)
        if(config.has_section(CONFIG_SECTION) and config.has_option(CONFIG_SECTION, 'region')):
            result = config.get(CONFIG_SECTION, 'region')
    return result


def get_credentials():
    """
    Returns the AWS credential pairs stored in the environment variables:
    AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY.
    """
    if (not 'AWS_ACCESS_KEY_ID' in os.environ):
        raise KeyError('No access key found. Please set the environment variable AWS_ACCESS_KEY_ID, or using graphlab.aws.set_credential()')
    if (not 'AWS_SECRET_ACCESS_KEY' in os.environ):
        raise KeyError('No secret key found. Please set the environment variable AWS_SECRET_ACCESS_KEY, or using graphlab.aws.set_credential()')
    return (os.environ['AWS_ACCESS_KEY_ID'], os.environ['AWS_SECRET_ACCESS_KEY'])


def launch_EC2(instance_type=DEFAULT_INSTANCE_TYPE, region=None, CIDR_rule=None, auth_token=None,
               security_group=None, tags=None, use_secure_connection=True):
    """
    Launches an EC2 instance of the specified type. It is expected that AWS credentials are already set
    as environment variables (see :py:func:`~.graphlab.aws.set_aws_credential()`), or configuration is provided in the Boto config file.

    Parameters
    ----------
    instance_type : string, optional
        The EC2 instance type to launch, default is m3.xlarge. We support all instance types
        except: 't2.micro', 't2.small', 't2.medium' and 'm3.medium'. For a list of instance_types,
        please refer to `here <http://aws.amazon.com/ec2/instance-types/#Instance_Types>`_.

    region : string, optional
        The AWS region in which to launch your instance. Default is 'us-west-2'.

    CIDR_rule : string or list of strings, optional
        The Classless Inter-Domain Routing rule(s) to use for the instance. Useful for restricting the IP
        Address Range for a client. Default is no restriction. If you specify CIDR_rule(s), you must also
        specify a security group to use.

    auth_token : string, optional
        The Authentication Token to be used by the instance. By default a randomly generated token is used.

    security_group : string, optional
        The name of the security group for the EC2 instance to use.

    tags : dict, optional
        A dictionary containing the name/value tag pairs to be assigned to the instance. If you want
        to create only a tag name, the value for that tag should be the empty string (i.e. ''). 
        In addition to these specified tags, a 'GraphLab' tag will also be assigned. 

    use_secure_connection : boolean, optional
       Determine whether the communication, between your computer and the EC2 instance, should be encrypted. Default is True.
    """
    # Check existing connection
    if glconnect.is_connected():
        __LOGGER__.warning('You have GraphLab objects instantiated on the current server. If you want to \n'
                           'launch a new server, you must first stop the current server connection by running: \n'
                           '\'graphlab.connect.main.stop()\'. Be warned: you will lose these instantiated objects.')
        return

    assert not glconnect.is_connected()

    (server_public_key, server_secret_key, client_public_key, client_secret_key) = ("", "", "", "")
    if use_secure_connection:
        (server_public_key, server_secret_key) = get_public_secret_key_pair()
        (client_public_key, client_secret_key) = get_public_secret_key_pair()
        __LOGGER__.debug('Launching server with public key: %s' % server_public_key)

    server = _ec2_factory(instance_type=instance_type, region=region, CIDR_rule=CIDR_rule, auth_token=auth_token,
                          security_group=security_group, tags=tags, server_public_key=server_public_key,
                          server_secret_key=server_secret_key)

    try:
        # Once the EC2 instance starts up, the client still need to wait for graphlab engine to start up.
        # So allow for a larger than normal number of ping failures. This is especially important when launching
        # in other AWS regions (since longer latency and longer download from S3 for engine binary)
        server.start(num_tolerable_ping_failures=120)
    except Exception as e:
        __LOGGER__.error("Unable to successfully connect to GraphLab Server on EC2 instance: '%s'. Please check AWS Console to make sure any EC2 instances launched have been terminated." % e)
        server.stop()
        raise e

    num_tolerable_ping_failures = 3
    client = Client([], server.get_server_addr(), num_tolerable_ping_failures, public_key=client_public_key,
                    secret_key=client_secret_key, server_public_key=server_public_key)
    auth_token = server.get_auth_token()
    if auth_token:
        client.add_auth_method_token(auth_token)
    client.start()
    glconnect._assign_server_and_client(server, client)


def set_credentials(access_key_id, secret_access_key):
    """
    Set the AWS credential environment variables:
    AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY.

    Parameters
    ----------
    access_key_id : string
        Value of AWS_ACCESS_KEY_ID

    secret_access_key : string
        Value of AWS_SECRET_ACCESS_KEY
    """
    os.environ['AWS_ACCESS_KEY_ID'] = str(access_key_id)
    os.environ['AWS_SECRET_ACCESS_KEY'] = str(secret_access_key)


@__catch_and_log__
def list_instances(region='us-west-2'):
    """
    Returns GraphLab EC2 instance that are currently running.

    Parameters
    ----------
    region : string, optional
        The AWS region in which to list instances. Default is 'us-west-2'.
    """
    __LOGGER__.info("Listing instances in %s." % region)
    aws_connection = boto.ec2.connect_to_region(region)
    recent_ids = [i.res_id for i in aws_connection.get_all_tags(filters={'tag:GraphLab': ''})]
    active_instnaces = aws_connection.get_all_instance_status(instance_ids=recent_ids)
    return [i.id for i in active_instnaces]


@__catch_and_log__
def status():
    """
    Returns the status of any running EC2 instance associated with the current client.
    """
    server = glconnect.get_server()
    if server is None or not isinstance(server, _EC2):
        return 'No instance running.'
    aws_connection = boto.ec2.connect_to_region(server.region)
    status = aws_connection.get_all_instance_status(instance_ids=[server.instance_id])
    __LOGGER__.debug("EC2 status returned: %s" % status)
    if(not(len(status) == 1)):
        __LOGGER__.error('Invalid response from EC2. Unable to determine EC2 status. Please consult '
                         'the AWS EC2 Console.')

    return status[0].state_name


def terminate_EC2():
    """
    Brings down the current EC2 instance.
    """
    server = glconnect.get_server()
    if server is not None and isinstance(server, _EC2):
        __LOGGER__.debug("Stopping EC2 instance.")
        glconnect.stop()
    else:
        __LOGGER__.info("No EC2 instance to stop.")


def _ec2_factory(instance_type, region=None, CIDR_rule=None, auth_token=None, security_group=None,
                 tags=None, server_public_key='', server_secret_key='', user_data = {},
                 credentials = {}):

    # get and validate the product / registration key
    try:
        product_key = graphlab.product_key.get_product_key()
    except KeyError as k:
        __LOGGER__.error(k.message)
        return

    if (not graphlab.product_key.is_product_key_valid(product_key)):
        __LOGGER__.error("Product Key validation failed, please confirm your product key is correct. "
                         "If you believe this key to be valid, please contact support@graphlab.com")
        return

    # Determine region
    if(region is None):
        region = _get_region_from_config()
        if(region is None):
            region = 'us-west-2'

    # If a CIDR rule is specified, a security group must also be specified.
    if CIDR_rule is not None and security_group is None:
        __LOGGER__.error('If you specify a CIDR rule, you must also specify an existing security group.')
        return
    # If no security group is specified (and no CIDR rule), use a default value for the security group name.
    elif security_group is None:
        security_group = GRAPHLAB_NAME

    if CIDR_rule is None:
        CIDR_rule = DEFAULT_CIDR_RULE

    # By default the EC2 instance will use its instance id as the auth_token.
    try:
        server = _EC2(instance_type, region, CIDR_rule, auth_token, product_key, security_group,
                      tags, server_public_key, server_secret_key, user_data, credentials)    # Launches the EC2 instance
    except Exception as e:
      __LOGGER__.error("Unable to launch EC2 instance: '%s'. Please check AWS Console to make sure any EC2 instances launched have been terminated." % e)
      # since EC2 launch could be part of a script, we want to raise the exception so that remaining code
      # stops executing (otherwise a local server would be created and the code would continue locally)
      raise e

    return server
