import tempfile as _tempfile
import shutil as _shutil
import imp
import json as _json
import datetime as _datetime
import os as _os
import requests as _requests
import shutil as _shutil
import subprocess as _subprocess
import time as _time
import abc as _abc 
import sys as _sys
import logging
import traceback
import json

from _context import ENVIRONMENT
import _timer as _timer

from boto import connect_s3 as _connect_s3
from boto.s3.key import Key as _s3_key

import graphlab
from graphlab_util.config import DEFAULT_CONFIG as _gc
from graphlab.connect.aws._ec2 import _ec2_factory, get_credentials as _get_credentials
import _artifact as _artifact
import _job as _job

from _hadoop_wrap import hadoop_setup

__LOGGER__ = logging.getLogger(__name__)


class ExecutionEnvironment(object):
    """
    Base class for all environments. Each derived class expected to implement run method.
    """

    __metaclass__ = _abc.ABCMeta

    def __init__(self, name):
        self.name = name

    @_abc.abstractmethod
    def run_job(self, job, session=None):
        raise NotImplementedError("Abstract Base Class")

    def run(self, session, tasks, name, environment):
        """
        Runs the given job in the environment.

        Parameters
        ----------
        session : Session
            Session to this run is bound to.
        tasks : List of tasks
            Cloned tasks to execute, in sequence
        name : string
            Name of this execution, should be the name of the Job returned
        environment : Environment
            Configuration for this execution. The type must match the ExecutionEnvironment.

        Returns
        -------
        Job object that should be queried for results.
        """
        job = _job.Job(name, tasks=tasks, environment=environment)
        return self.run_job(job, session)


class LocalExecutionEnvironment(ExecutionEnvironment):
    """
    Execute on local machine, synchronously.
    """

    def __init__(self):
        super(LocalExecutionEnvironment, self).__init__(ENVIRONMENT.Local)

    def run_job(self, job, session):
        job._status = 'Running'
        try:

            job = self._run(job, session)
            job._status = 'Completed'
        
        except Exception as e:
            job._status = 'Failed'
            raise e

        if session is not None:
            session.save(job)

        return job

    def run_task(self, task, metrics, location=None):
        # special-case Job inside Job
        if isinstance(task, _job.Job):
            run = self.run_job(task, None)
            metrics.update(run.metrics)
            return

        # run a task instead
        if not isinstance(task, _artifact.Task):
            raise Exception("Cannot run this type of task")
        if task.get_code() is None or task.get_code() == "":
            raise Exception("No code to run, artifact not fully initialized")

        starttime = str(_datetime.datetime.now())
        __LOGGER__.info("Task started: %s" % task.get_name())

        # path for execution
        if location is None:
            location = _tempfile.mkdtemp(prefix='task-exec-')
        path = location
        curdir = _os.path.abspath(_os.path.curdir)
        runtime_task = _artifact.RuntimeTask(task)
        try:
            _os.chdir(path)
            __LOGGER__.debug("Path for execution: %s" % path)
            
            code = task._get_runnable_code()
            if isinstance(code, dict):
                # Code was specified as a file and function name
                
                # unpack files into execution path!
                for filename in code.keys():
                    with open(_os.path.join(path, filename), 'w') as f:
                        f.write(code[filename])

                modulename = task._data['filename']
                func_name = task._data['funcname']
                
                # add current path to system path so module loading succeeds
                _sys.path.append(path)

                module = imp.load_source('', modulename)
                code = getattr(module, func_name)

            try:
                # now actually to run the code
                with _timer.Timer() as t:
                    code(runtime_task)
            except Exception as e:
                __LOGGER__.error("Exception raised from task: '%s' code: %s" % (task.name, e))
                metrics['task.%s.error' % task.name] = True
                metrics['task.%s.error_msg' % task.name] = e
                metrics['task.%s.traceback' % task.name] = traceback.format_exc(1)
                raise RuntimeError("Unable to complete task successfully, Exception raised, trace: %s" % 
                        traceback.format_exc(0), e)

            # write out outputs to disk, if outputs specified
            for (name, value) in task.get_outputs().iteritems():
                if value is not None and task.outputs[name] is not None:
                    __LOGGER__.debug("Output named: '%s' written to location: '%s'" % (name, value))
                    task.outputs[name].save(value)

            __LOGGER__.info("Task completed: %s" % task.get_name())

            metrics['task.%s.last_run' % task.name] = starttime
            metrics['task.%s.runtime' % task.name] = t.secs
            metrics['task.%s.metrics' % task.name] = runtime_task.metrics

        finally:
            if path in _sys.path:
                _sys.path.remove(path)
            _os.chdir(curdir)

        return runtime_task

    def _run(self, job, session=None):
        if not job.metrics:
            job.metrics = {}
        job.metrics.clear()

        __LOGGER__.info("Execution started: %s" % job.name)

        exec_dir = _tempfile.mkdtemp(prefix='job-exec-')
        __LOGGER__.info("Execution path: %s" % exec_dir)

        self._completed = {}
        
        # Nothing done in parallel.
        for cur_step in job._sequence:
            for task in cur_step:
                task.env = self
                self.run_task(task, job.metrics, location=exec_dir)
                if session is not None:
                    session.save(job)
                self._completed[task.name] = task

        __LOGGER__.info("Execution completed: %s" % job.name)
    
        if session is not None:
            session.save(job, typename=str(_job.Job.__name__))
        return job

    def get_dependent_task(self, name):
        """
        Return the dependent task by name.

        Chooses the most recently run task from this job and returns the object from its execution.

        Returns
        -------
        task instance for dependency, if not found then None
        """
        if name is None:
            return None

        dep = None
        if name in self._completed:
            dep = self._completed[name]
        
        __LOGGER__.debug("Satisfied dependency named: '%s' with task: %s" % (name, dep))
        return dep


class Ec2ExecutionEnvironment(ExecutionEnvironment):

    def __init__(self):
        super(Ec2ExecutionEnvironment, self).__init__(ENVIRONMENT.EC2)

    @staticmethod
    def _start_commander_host(environment, credentials):
        user_data = {
            'daemon': True, 'num_children_host': environment.num_hosts - 1,
            'aws_access_key': credentials['aws_access_key_id'],
            'aws_secret_key': credentials['aws_secret_access_key']
            }

        if('GRAPHLAB_TEST_AMI_ID' in _os.environ and 'GRAPHLAB_TEST_ENGINE_URL' in _os.environ):
            user_data['GRAPHLAB_TEST_AMI_ID'] = _os.environ['GRAPHLAB_TEST_AMI_ID']
            user_data['GRAPHLAB_TEST_ENGINE_URL'] = _os.environ['GRAPHLAB_TEST_ENGINE_URL']
        if('GRAPHLAB_TEST_EC2_KEYPAIR' in _os.environ):
            user_data['GRAPHLAB_TEST_EC2_KEYPAIR'] = _os.environ['GRAPHLAB_TEST_EC2_KEYPAIR']

        # Launch the (commander) EC2 host
        commander = _ec2_factory(environment.instance_type, region = environment.region,
                                 CIDR_rule = environment.CIDR_rule,
                                 security_group = environment.security_group,
                                 tags = environment.tags, user_data = user_data,
                                 credentials = credentials)
        return commander

    def run_job(self, job, session=None):
        # Determine which AWS credentials to use. 
        aws_access_key, aws_secret_key = None, None
        if(hasattr(job.environment, 'aws_access_key') and hasattr(job.environment, 'aws_secret_key')):
            aws_access_key = job.environment.aws_access_key
            aws_secret_key = job.environment.aws_secret_key
        else: 
            aws_access_key, aws_secret_key = _get_credentials()
        credentials = {'aws_access_key_id': aws_access_key, 'aws_secret_access_key': aws_secret_key}

        s3_log_folder = job.environment.s3_log_folder_path + job.name + '-' + str(_time.time()) + '/'
        s3_state_key = s3_log_folder + 'status'
        s3_log_key = s3_log_folder + 'log.txt'

        # Make sure we can write to S3
        try:
            conn = _connect_s3(**credentials)
            bucket = conn.get_bucket(job.environment.s3_bucket)
            k = _s3_key(bucket)
            k.key = s3_state_key
            k.set_contents_from_string('Pending')
        except:
            raise RuntimeError("Unable to start job. Could not write to S3 bucket: '%s'"
                               % job.environment.s3_bucket)

        serialized_job_folder = _tempfile.mkdtemp(prefix='graphlab-ec2-job')
        serialized_job_file_path = _os.path.join(serialized_job_folder, 'job')
        job._serialize(serialized_job_file_path)

        commander = Ec2ExecutionEnvironment._start_commander_host(job.environment, credentials)
        post_url = "http://%s:9004/submit" % commander.public_dns_name
        __LOGGER__.debug("Sending %s to %s" % (serialized_job_file_path, post_url))

        # Swallow all logging from the 'requests' module.
        logging.getLogger('requests').setLevel(logging.CRITICAL)

        # Technically the host should be started up now, but more than likely it needs a bit longer
        # before it's really started up.
        MAX_TRIES, current_tries, response = 60, 0, None
        post_params = {'s3_bucket': job.environment.s3_bucket,
                  's3_state_key': s3_state_key, 
                  's3_log_key': s3_log_key,
                  'required_packages': job.get_required_packages()
                  }
        while current_tries <  MAX_TRIES:
            with open(serialized_job_file_path, 'rb') as job_file:
                files = {'file': job_file}
                try: 
                    # Try to submit the work.
                    current_tries += 1
                    response = _requests.post(post_url, files=files, params=post_params)
                    break   # Success
                except _requests.exceptions.ConnectionError, e:
                    if current_tries >= MAX_TRIES:
                        commander.stop()
                        raise RuntimeError("Unable to submit job to EC2 instance: '%s'. Please check AWS Console to make sure any EC2 instances launched have been terminated." % commander.instance_id)
            _time.sleep(2)

        # Check the response from the commander.
        if any([not response, not hasattr(response, 'status_code'), response.status_code != 200]):
            commander.stop()
            try:
                response = _json.loads(response.text)
                assert(response['Exception']) # Will throw exception if key not present or value is None.
            except:
                raise RuntimeError("EC2 instance could not start the job.")
            else:
                raise RuntimeError("EC2 instance could not start the job. Exception: %s"
                                   % str(response['Exception']))

        # delete previous job definition since this one is the 'right' one
        if graphlab.deploy.jobs[job.name] is not None:
            __LOGGER__.debug("Deleting previous job name since more specific type of Job being created/returned: %s" % job.name)
            graphlab.deploy.jobs.delete(job.name)

        result = _job.Ec2Job(job.name, job.tasks, job.environment, commander.public_dns_name,
                             commander.instance_id, job.environment.s3_bucket, s3_state_key,
                             s3_log_key)

        if session is not None:
            session.save(result)
        __LOGGER__.info('Job successfully submitted for execution to EC2 and execution has begun.')
        _shutil.rmtree(serialized_job_folder)

        return result


class HadoopExecutionEnvironment(ExecutionEnvironment):

    def __init__(self):
        super(HadoopExecutionEnvironment, self).__init__(ENVIRONMENT.Hadoop)
        self.hadoop_jar = _os.path.join(_os.path.dirname(_os.path.realpath(__file__)), "graphlabHadoopYarn-1.0.jar")


    def run_job(self, job, session=None):
        """
        call the hadoop client but pass in the hadoop_wrap file to actually execute stuff
        use path as the resource_dir

        hadoop jar /tmp/graphlabHadoopYarn-1.0.jar Client -debug -jar /tmp/graphlabHadoopYarn-1.0.jar -container_memory 2096
            -debug -python_script /blah/hadoop_wrap.py
            -resource_dir /blah -product_key /home/cloudera/.graphlab/config
           	-cache_virtualenv , -gl_source < none|| native || hdfs://blah:8020/gl.tar.gz >

        """
        try:
            # call hadoop_wrap.setup
            memory = job.environment.get_container_memory()
            path = _tempfile.mkdtemp(prefix='hadoop-exec-dir-')
            cmd_args = hadoop_setup(path, job, memory)


            # format hadoop command
            if job.environment.get_config_dir() is not None:
                hadoop_base = "hadoop --config %s " % job.environment.get_config_dir()
            else:
                hadoop_base = "hadoop "

            egg_path = job.environment.get_graphlab_source()
            __LOGGER__.debug("session=%s \n task=%s" % (session, job))
            __LOGGER__.debug("path for egg: %s" % egg_path)
            source = self.hadoop_jar
            hadoop_cmd = "%s jar %s Client -jar %s -gl_source %s" % \
                         (hadoop_base, source, source, egg_path)

            reqs = job.get_required_packages()
            if reqs:
                reqs_pip = path + "/user_requirements.pip"
                with open(reqs_pip, 'w') as f:
                    for line in reqs:
                        f.write(line +'\n')

                cmd_args['dependencies'] = reqs_pip


            for k, v in cmd_args.items():
                hadoop_cmd += " -%s %s" % (k, v)

            hadoop_cmd += ' -container_vcores %s ' \
                          % job.environment.get_virtual_cores()
            hadoop_cmd += " -gl_artifact "
            hadoop_cmd += ' -json '
            if _gc.mode != 'PROD':
                hadoop_cmd += ' -debug '

            if 'GRAPHLAB_HADOOP_CACHE_VIRTUALENV' in _os.environ:
                hadoop_cmd += " -cache_virtualenv"

            # call hadoop command
            __LOGGER__.info("Submitting job to Hadoop cluster using command= \n%s" % hadoop_cmd)
            proc = _subprocess.Popen(hadoop_cmd, shell=True,
                                              stderr=_subprocess.STDOUT,
                                              stdout=_subprocess.PIPE )

            __LOGGER__.debug("Submitting job to cluster")

            app_id = None
            AM_state = None
            app_state = None
            start_time = None
            end_time = None
            for line in proc.stdout:
                if _job.HadoopJob._json_flag in line:
                    ## json line looks like TAGjsonTAG
                    clean = line.split(_job.HadoopJob._json_flag)
                    app_report=json.loads(clean[1])
                    app_id = app_report['AppId']
                    AM_state = app_report['AppState']
                    app_state = app_report['DistributedFinalState']
                    start_time = app_report['StartTime']
                    end_time = app_report['FinishTime']
                    __LOGGER__.info('got application report=%s\n' % app_report)
                    __LOGGER__.info( "app_id=%s" % app_id)
                    __LOGGER__.info("AM_state=%s" % AM_state)
                    __LOGGER__.info("app_state=%s"  % app_state)

                else:
                    __LOGGER__.debug(line.strip('\n'))

            if not app_id:
                raise RuntimeError("Error submitting application or determining application id. Please confirm that you can correctly access the Hadoop environment specified (try running the above command from a terminal to see more diagnostic output).")
            
            # delete previous job definition since this one is the 'right' one
            if graphlab.deploy.jobs[job.name] is not None:
                __LOGGER__.debug("Deleting previous job name since more specific type of Job being created/returned: %s" % job.name)
                graphlab.deploy.jobs.delete(job.name)

            job = _job.HadoopJob(job.name, job.tasks, job.environment, app_id)
            job._app_state = app_state
            job._yarn_AM_state = AM_state
            job._start_time = start_time
            job._end_time = end_time

        finally:
            # make sure to remove temp directory created
            _shutil.rmtree(path)

        return job


class LocalAsynchronousEnvironment(ExecutionEnvironment):

    def __init__(self):
        super(LocalAsynchronousEnvironment, self).__init__(ENVIRONMENT.LocalAsynchronous)

    def run_job(self, job, session=None):
        """
        Runs a job asynchronously in a background process.

        1. Create a /tmp directory for this execution
        2. Serialize the Job to disk so can be read by other process
        3. Start additional process
        4. Return LocalAsynchronousJob object to caller
        """
        # process launch with a pickled Job as file path and session.location
        driver_file_path = _os.path.join(_os.path.dirname(__file__), '_graphlabJob.py')
        path = _tempfile.mkdtemp(prefix='graphlab-async-')
        job_path = _os.path.join(path, 'job-definition')
        job._serialize(job_path)

        env = _os.environ.copy()
        log_file_path = _os.path.join(path, 'execution.log')
        log_file = open(log_file_path, 'w')

        # write error exitcode at launch, so if process is gone we know it failed
        with(open(_os.path.join(path, 'exitcode'), 'w')) as f:
            f.write('1')

        arglist = ['python', driver_file_path, job_path]

        # Launch the other process
        __LOGGER__.debug("Launching process with arglist: %s" % arglist)
        proc = _subprocess.Popen(arglist, env=env, stdin=_subprocess.PIPE, stdout=log_file,
                                 stderr=_subprocess.STDOUT, bufsize=-1, preexec_fn=lambda: _os.setpgrp())

        __LOGGER__.debug("Process launched with pid: %d" % proc.pid)
        
        # delete previous job definition since this one is the 'right' one
        if graphlab.deploy.jobs[job.name] is not None:
            __LOGGER__.debug("Deleting previous job name since more specific type of Job being created/returned: %s" % job.name)
            graphlab.deploy.jobs.delete(job.name)
        
        return _job.LocalAsynchronousJob(proc, job.name, log_file_path, tasks=job.tasks,
                                         environment=job.environment)

# register with base class that these two classes implement the base class
ExecutionEnvironment.register(LocalExecutionEnvironment)
ExecutionEnvironment.register(Ec2ExecutionEnvironment)
ExecutionEnvironment.register(HadoopExecutionEnvironment)
ExecutionEnvironment.register(LocalAsynchronousEnvironment)
