import __builtin__
import os as _os
import pickle as _pickle
import re as _re
import fnmatch as _fnmatch
import tempfile as _tempfile
import time as _time
import collections
import logging

import graphlab.cython.cloudpickle as _cloudpickle
import _artifact as _artifact
import _context as _ctx
import _executionenvironment as _env
import _job as _job

import graphlab as _gl


__LOGGER__ = logging.getLogger(__name__)


class Session(object):
    """
    Manages Tasks, Environments, and Jobs. A session object is an abstraction that provides functionality
    manage the lifecycle of: Tasks, Environments, and Jobs. Session is not intended to be interacted with directly, 
    instead use graphlab.deploy.tasks, graphlab.deploy.jobs, graphlab.deploy.environments.
    """

    def __init__(self, location):
        """
        Create a new session, specifying a directory to use for the session. This directory will be used for
        saving and loading all Tasks, Pipelines, Contexts, and Jobs. If the directory does not exist it will be
        created.
        """

        if location is None:
            raise Exception('Location is a required parameter.')

        self.location = location
        self._cache = {}
        self._objects = {}

        if _os.path.exists(self.location):
            if _os.path.isfile(self.location):
                raise Exception('Illegal location for session, must be a directory.')
        else:
            try:
                _os.makedirs(self.location)
            except OSError as e:
                self.location = _tempfile.mkdtemp(prefix='graphlab-tmp-session-')
                if not _os.path.exists(self.location):
                    _os.makedirs(self.location)
                __LOGGER__.warning("Unable to create session in specified location: '%s', using tmp location instead: '%s'" % (location, self.location))

    def register(self, artifact):
        """
        Objects that should be managed by this session are required to register with the session.

        Parameters
        ----------
        artifact : Task | Environment | Job
            Task or Environment instance to be managed by this sesssion. If another Task or Environment has 
            the same name, then this new object will overwrite the previous object.

        Returns
        -------
        True if successfully registered, False otherwise
        """
        if artifact is None:
            return False

        if self._cache == {} or self._objects == {}:
            self.list(refresh=True, typename=None)
        
        if artifact.name is None or artifact.name == "":
            __LOGGER__.warning("Name is required for objects managed by a session.")
            return False

        if artifact.name in self._objects or artifact.name in self._cache:
            raise NameError("An item with the name '%s' already exists, please rename." % artifact.name)

        self._objects[artifact.name] = artifact
        return True

    def list(self, refresh=False, typename=None):
        """
        List all items managed by this session. Items include Tasks, Environments, and Jobs.

        Parameters
        ----------
        refresh : boolean
            Refresh the listing, meaning re-query location to get current artifacts.
        typename : str (optional)
            Optionally limit the output a specific type (Task, Environment, Job)

        Returns
        -------
        SFrame with columns for name, type, Modified?, In-Memory?, Unsaved?
        """
        if typename is None or refresh is True:
            files = _os.listdir(self.location)
            catalog = {}
            for f in files:
                try:
                    (filename, ext) = _os.path.splitext(f)
                    self._open(filename, catalog, typename=ext[1:])
                except Exception as e:
                    __LOGGER__.warning('Error loading file: %s, error: %s' % (f, e))

            for artifact in catalog.itervalues():
                status = Session._get_status_from_artifact(artifact)
                self._cache[artifact.name] = (artifact.name, __builtin__.type(artifact), status)

            catalog.clear()

        for artifact, name in enumerate(self._objects):
            artifact = self._objects[name]
            status = Session._get_status_from_artifact(artifact)
            self._cache[name] = (name, __builtin__.type(artifact), status)

        if typename is not None:

            if typename == 'Job':
                required_type = _job.Job
            elif typename == 'Environment':
                required_type = _ctx.Environment
            elif typename == 'Task':
                required_type = _artifact.Task
            else:
                raise Exception("Unknown typename: %s" % typename)
            
            # filter out by typename
            # filtered_values = [ (n, t, status, in_memory) 
            # for (n, t, status, in_memory) in self._cache.values() if True] # issubclass(t, required_type)]
            filtered_values = [ (n, t, status) 
                    for (n, t, status) in self._cache.values() if issubclass(t, required_type)]
        else:
            filtered_values = self._cache.values()

        names = _gl.SArray(data=[n for (n, t, status) in filtered_values])
        types = _gl.SArray(data=[str(t.__name__) for (n, t, status) in filtered_values])
        statuses = _gl.SArray(data=[str(status) for (n, t, status) in filtered_values])

        sf = _gl.SFrame()
        sf['Name'] = names
        sf['Type'] = types
        sf['Unsaved changes?'] = statuses
        sf.add_row_number(column_name='Index', start=0)

        return sf

    @staticmethod
    def _get_status_from_artifact(artifact):
        if hasattr(artifact, '_modified_since_last_saved'):
            status = artifact._modified_since_last_saved == None or artifact._modified_since_last_saved is True
            if status is True:
                return 'Yes'
            else:
                return 'No'
        return 'No'

    def _delete(self, artifact=None, pattern=None, typename=None):
        """
        Delete artifact from disk. This will not ask permission and will delete
        the task/pipeline, so please be careful using this.

        Parameters
        ----------
        artifact : object | string
            Artifact to delete, passed as an object or by name as a string
        pattern : string
            Spec to delete artifacts ex. "*-Local-*". This uses fnmatch.fnmatch
        typename : str (optional)
            Optionally limit the output a specific type (Task, Environment, Job)
        """
        loaddir = self.location
        if pattern is not None:
            files = _os.listdir(loaddir)
            for file in _fnmatch.filter(files, pattern):
                try:

                    # skip file if not type specified
                    if typename is not None:
                        (root, ext) = _os.path.splitext(file)
                        
                        if typename == 'Job':
                            required_type = _job.Job
                        elif typename == 'Environment':
                            required_type = _ctx.Environment
                        elif typename == 'Task':
                            required_type = _artifact.Task
                        else:
                            __LOGGER__.warning("Unknown typename of file: %s" % typename)

                        if ext[1:] != str(required_type.__name__):
                            continue

                    __LOGGER__.info("Deleting file: %s" % file)
                    artifact = self.__open_artifact(_os.path.join(loaddir, file), {})
                    self.__remove_artifact(artifact.name, typename=typename)
                except Exception as e:
                    __LOGGER__.warning("Error in removing, deleted file instead: %s, error: %s" % (file, e))
                    # invalidate cache since error deleting, go to disk for future
                    self._cache = {}
                    _os.remove(_os.path.join(loaddir, file))
        else:
            if isinstance(artifact, str):
                self.__remove_artifact(artifact, typename)
            elif hasattr(artifact, "name"):
                self.__remove_artifact(artifact.name, typename)
            elif isinstance(artifact, int):
                name = self._get_name_from_index(artifact, typename)
                if name is None:
                    raise IndexError("Unable to find item to delete by index provided, nothing deleted. Index invalid: %d" % artifact)
                self.__remove_artifact(name, typename)
            else:
                raise TypeError("Invalid parameters to delete, see documentation.")

    def __remove_artifact(self, name, typename):
        
        if typename == 'Job':
            required_type = _job.Job
        elif typename == 'Environment':
            required_type = _ctx.Environment
        elif typename == 'Task':
            required_type = _artifact.Task
        else:
            raise TypeError("Invalid typename parameter")

        found = False
        filename = self._get_filename_from_name(name) + "." + str(required_type.__name__)
        path = _os.path.join(self.location, filename)
        __LOGGER__.debug("Removed: %s" % path)
        if _os.path.exists(path) and _os.path.isfile(path):
            found = True
            _os.remove(path)
        
        # remove from both caches
        if name in self._cache:
            found = True
            del self._cache[name]
        if name in self._objects:
            found = True
            del self._objects[name]

        if found is False:
            raise KeyError("Unable to find item named: '%s', nothing deleted." % name)

    def _get_name_from_index(self, index, typename):
        output_sf = self.list(refresh=True, typename=typename)
        
        if index >= output_sf.num_rows():
            return None
        
        return output_sf['Name'][index]

    def _load(self, name, typename=None):
        """
        Load an item by name.

        Parameters
        ----------
        name : string
            Name of item to load, full name required (ex. s0)
        typename : str (optional)
            Optionally limit the output a specific type (Task, Environment, Job)

        Returns
        -------
        Instance of the item.
        """
        if isinstance(name, int):
            # index lookup instead
            name = self._get_name_from_index(name, typename)
            if name is None:
                return None

        # if already in self._objects, then go with that instead
        if name in self._objects:
            obj = self._objects[name]
            if isinstance(obj, _job.Job) and typename == 'Job':
                return obj
            elif isinstance(obj, _ctx.Environment) and typename == 'Environment':
                return obj
            elif isinstance(obj, _artifact.Task) and typename == 'Task':
                return obj

        return self._open(name, self._objects, typename=typename)

    def save(self, obj, typename=None):
        """
        Save the item to this session.

        Parameters
        ----------
        obj : object
            Object to save to this session
        typename : str (optional)
            Specify the type of this object (Task, Environment, Job)
        """
        if isinstance(obj, str):
            if obj in self._objects:
                obj = self._objects[obj]
            else:
                raise Exception("Unable to find artifact to save.")

        if not Session._is_known_type(obj):
            raise Exception("Trying to save an unknown type")

        savedir = self.location
        if typename is None:
            if isinstance(obj, _job.Job):
                typename = 'Job'
            elif isinstance(obj, _ctx.Environment):
                typename = 'Environment'
            elif isinstance(obj, _artifact.Task):
                typename = 'Task'
            else:
                __LOGGER__.error("Trying to save an unrecognized item of type: %s, saving failed." % type(obj))
                return

        filename = self._get_filename_from_name(obj.name) + "." + typename

        try:

            with __builtin__.open(str(_os.path.join(savedir, filename)), "w") as f:
                _cloudpickle.dump(obj, f)

            if hasattr(obj, '_modified_since_last_saved'):
                obj._modified_since_last_saved = False

        except Exception as e:
            __LOGGER__.warning("Error saving %s: '%s'" % (typename, e))

    def is_name_unique(self, name, typename=None):
        """
        Helper function to verify if a proposed artifact name is unique for this session.

        Parameters
        ----------
        name : str
            Name for artifact, confirm if unique or not
        typename : str (optional)
            Specify the type of this object (Task, Environment, Job)

        Returns
        -------
        True if unique, False otherwise.
        """
        filename = self._get_filename_from_name(name)
        if typename is not None:
            filename = "%s.%s" % (filename, typename)

        if _os.path.exists(_os.path.join(self.location, filename)):
            return False
        else:
            return True

    def __getitem__(self, item):
        return self._load(item)

    def __delitem__(self, key):
        return self._delete(key)

    @staticmethod
    def _is_known_type(obj):
        """
        Helper method that returns True when type is supported by Session.
        """
        return any([isinstance(obj, _artifact.Task), isinstance(obj, _artifact.Pipeline), 
            isinstance(obj, _ctx.Environment), isinstance(obj, _job.Job)])

    def __repr__(self):
        return self.__str__(typename=None)

    def __str__(self):
        return self.__str__(typename=None)

    def __str__(self, typename=None):
        if typename is None:
            ret = "\nSession (location: %s)\n\n" % (_os.path.abspath(self.location))
        else:
            ret = "\n%s(s): \n" % typename
        
        output_sf = self.list(refresh=True, typename=typename)
        if output_sf.num_rows() == 0:
            ret += '\nNothing to display.\n'
        else:
            rows_of_table = output_sf.__get_pretty_tables__(max_column_width=60)
            ret += '\n'.join([str(tb) for tb in rows_of_table])

        return ret

    def _open(self, name, catalog, check_cache=False, typename=None):
        """
        Instantiate named artifact. Store a reference in the catalog dictionary.
        """

        if check_cache is True and name in self._objects:
            return self._objects[name]

        loaddir = self.location

        if typename is not None:
            name = name + '.' + typename

        path = _os.path.join(loaddir, self._get_filename_from_name(name))

        if _os.path.exists(path) and _os.path.isfile(path):
            return self.__open_artifact(path, catalog)
        return None

    def __open_artifact(self, path, catalog):
        try:
            with __builtin__.open(path, 'r') as f:
                artifact = _pickle.load(f)
                if not Session._is_known_type(artifact):
                    raise Exception("Trying to load an unknown type")
                artifact._session = self
                if hasattr(artifact, "_modified_since_last_saved"):
                    artifact._modified_since_last_saved = False
                catalog[artifact.name] = artifact
                return artifact
        except Exception as e:
            __LOGGER__.error("Error opening from path: %s, error: %s" % (path, e))
            raise e

    def _get_filename_from_name(self, name):
        """Gets valid filename by renaming object"""

        if name is None or name == "":
            raise AttributeError('Missing name')

        name = str(name).strip().replace(' ', '_')
        return _re.sub(r'(?u)[^-\w.]', '', name)

def open(location=None):
    """
    Open and return an Artifact session.

    Parameters
    ----------
    location : os.path
        Disk location for this session, needs to be a directory. Default value
        is ~/.graphlab/artifacts.
    """
    if location is None:
        location = _os.path.expanduser(_os.path.join('~', '.graphlab', 'artifacts'))

    return Session(location)

class ScopedSession(object):
    """
    Allow one class to implement listing, loading, and deleting for Tasks, Jobs, Environments.
    
    This class is a helper over the Session object, restricting it to just one type (Task, Environment, etc).
    Used by graphlab.deploy.tasks, graphlab.deploy.environments to provide access to the session scoped by type.

    To load an item (in this example a Task):
    >>> gl.deploy.tasks # returns a listing of Task objects
    >>> task = gl.deploy.tasks['taskname'] # load Task and return it
    
    To delete an item (in this example a Job):
    >>> gl.deploy.jobs # returns a listing of Job objects
    >>> del gl.deploy.jobs['delete-me-named-job']
    
    """
    def __init__(self, session, type):
        self._session = session
        self._typename = type.__name__

    def __repr__(self):
        return self._session.__str__(typename=self._typename)

    def list(self, refresh=True):
        """
        List all items of this type (jobs, tasks, environments).

        Parameters
        ----------
        refresh : boolean
            Refresh the listing, meaning re-query location to get current artifacts.

        Returns
        -------
        SFrame with columns for name, type, Modified?, In-Memory?, Unsaved?
        """
        return self._session.list(refresh=refresh, typename=self._typename)

    def load(self, item):
        """
        Load an item by name.

        Parameters
        ----------
        name : string
            Name of item to load, full name required (ex. s0)
        
        Returns
        -------
        Instance of the item.
        """
        return self._session._load(item, typename=self._typename)

    def delete(self, item, pattern=None):
        """
        Delete an item from disk. This will not ask permission and will delete
        from the file system, so please be careful using this.

        Parameters
        ----------
        item : object | string
            Item to delete, passed as an object or by name as a string (Job, Task, Environment)
        pattern : string
            Spec to delete artifacts ex. "*-Local-*". This uses fnmatch.fnmatch
        """
        return self._session._delete(item, pattern=pattern, typename=self._typename)

    def __getitem__(self, item):
        return self._session._load(item, typename=self._typename)

    def __delitem__(self, key):
        self._session._delete(key, typename=self._typename)
        __LOGGER__.info("%s deleted successfully." % self._typename)
        return
