
from graphlab.data_structures.sgraph import SGraph
from graphlab.data_structures.sframe import SFrame
import graphlab.toolkits.main as main
from graphlab.toolkits.model import Model


class ConnectedComponentModel(Model):
    """
    A model object containing a componentid for each vertex and the total
    number of weakly connected component in the graph.

    An instance of this model can be created using :func:`graphlab.connected_components.create`.
    Do NOT construct the model directly.
    """
    def __init__(self, input_params, model):
        '''__init__(self)'''
        self.__proxy__ = model
        self._input = input_params
        self._fields = ['num_of_components', 'runtime', 'componentid', 'graph',
            'component_size']

    def list_fields(self):
        """
        List of fields stored in the model. Each of these fields can be queried
        using the ``get`` function.

        Returns
        -------
        out : list
            A list of fields that can be queried using the ``get`` method.
        """

        return self._fields

    def get(self, field):
        """
        Get the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained programmatically using the
        :func:`~graphlab.graph_analytics.ConnectedComponentModel.list_fields`
        method.

        Each of these fields can be queried in one of two ways:

        >>> out = m['field']
        >>> out = m.get('field')  # equivalent to previous line

        +-----------------+----------------------------------------------------+
        |      Field      | Description                                        |
        +=================+====================================================+
        | componentid     | SFrame with the component ID of each vertex        |
        +-----------------+----------------------------------------------------+
        | graph           | Input graph with component ID as a vertex property |
        +-----------------+----------------------------------------------------+
        | component_size  | An SFrame with the size of each component          |
        +-----------------+----------------------------------------------------+

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out : [various]
            The current value of the requested field.

       """

        if (field == 'componentid'):
            return SGraph(_proxy=self.__proxy__.get('__graph__')).get_vertices()
        elif (field == 'graph'):
            return SGraph(_proxy=self.__proxy__.get('__graph__'))
        elif (field == 'component_size'):
            return SFrame(_proxy=self.__proxy__.get('component_size'))
        else:
            return self.__proxy__.get(field)

    def summary(self):
        """Return a summary of the model including hyperparameters, and runtime information."""

        ret = {'hyperparams': self._input}
        for key in ['runtime', 'num_of_components']:
            ret[key] = self.get(key)
        return ret

    def _get_wrapper(self):
        return lambda m: ConnectedComponentModel(self._input, m)


def create(graph, verbose=True, plot=False):
    """
    Compute the number of weakly connected components in the graph. Return a
    model object with total number of weakly connected components as well as the
    component ID for each vertex in the graph.

    Parameters
    ----------
    graph : SGraph
        The graph on which to compute the triangle counts.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : Model
        Model object that contains the componentid for each vertex.

    Examples
    --------
    If given an :class:`~graphlab.SGraph` ``g``, we can create
    a :class:`~graphlab.connected_components.ConnectedComponentModel` as
    follows:

    >>> cc = connected_components.create(g)
    >>> print cc.summary()
    >>> print cc.list_fields()

    We can obtain the ``component id`` corresponding to each vertex in the
    graph ``g`` as follows:

    >>> cc_ids = cc['componentid']  # SFrame

    We can obtain a graph with additional information about the ``component
    id`` corresponding to each vertex as follows:

    >>> cc_graph = cc['graph']      # SGraph

    """
    if not isinstance(graph, SGraph):
        raise TypeError('graph input must be a SGraph object.')

    if plot is True:
        print "The plot functionality for connected components is not yet " + \
            "implemented."

    params = main.run('connected_component', {'graph': graph.__proxy__},
                      verbose, plot)
    return ConnectedComponentModel({}, params['model'])
