define('common_util',
      ['d3'],
      function(d3) {

  // from # rows to # milliseconds between requests
  var pollIntervalThresholds = [
    [Math.pow(2, 20), 1000],
    [Math.pow(2, 27), 5000]
  ];
  var pollIntervalScale = d3.scale.linear()
    .domain(pollIntervalThresholds.map(function(a) { return a[0]; }))
    .range(pollIntervalThresholds.map(function(a) { return a[1]; }));

  function assert(x) {
    if ('assert' in console) {
      console.assert(x);
    } else if (!x) {
      var msg = 'assertion failed';
      if ('error' in console) {
        console.error(msg);
      } else if ('log' in console) {
        console.log(msg);
      }
      if ('trace' in console) {
        console.trace();
      }
    }
  }

  function merge(x1, x2) {
    var ret = {};
    for (var attr in x1) {
      ret[attr] = x1[attr];
    }
    for (var attr in x2) {
      // clobber properties in x1 with x2
      ret[attr] = x2[attr];
    }
    return ret;
  }

  return {
    assert: assert,
    merge: merge,
    partial: function(obj, keys) {
      /* returns a partial copy of 'obj',
       * with only the keys in 'keys' */
      var ret = {};
      keys.forEach(function(key) {
        ret[key] = obj[key];
      });
      return ret;
    },
    objectMap: function(obj, callback) {
      // like jQuery $.map, but instead of producing an
      // array it produces an object with similar structure
      // to the input.
      var ret = {};
      for (var key in obj) {
        ret = merge(ret, callback(key, obj[key]));
      }
      return ret;
    },
    tryGetProperty: function _tryGetProperty(obj, propName, defaultValue) {
      // equivalent to null-expanding ?. operator
      // (see https://visualstudio.uservoice.com/forums/121579-visual-studio/suggestions/3990187-add-operator-to-c)
      if (typeof defaultValue === 'undefined') defaultValue = null;
      // == null checks for undefined as well
      if (propName == null) return defaultValue;
      if (obj == null) return defaultValue;
      if (typeof propName === 'string' || typeof propName === 'number') {
        if (!(propName in obj)) return defaultValue;
        return obj[propName];
      } else {
        assert(Array.isArray(propName));
        var currentPropName = propName.shift();
        if (propName.length === 0) {
          // array base case
          return _tryGetProperty(obj, currentPropName, defaultValue);
        } else {
          // array recursive case
          return _tryGetProperty(
            _tryGetProperty(obj, currentPropName),
            propName,
            defaultValue);
        }
      }
    },
    deepEquals: function _deepEquals(a, b) {
      // deep compare arbitrary JS objects
      if (typeof a !== typeof b) return false;
      if (Array.isArray(a) !== Array.isArray(b)) return false;
      if (Array.isArray(a)) {
        // array compare
        if (a.length !== b.length) return false;
        var ret = true;
        for (var i=0; i<a.length; i++) {
          ret = ret && _deepEquals(a[i], b[i]);
          if (!ret) return ret; // shortcut if false
        }
        return ret;
      }
      if (typeof a === 'object') {
        var keys = {}; // keep track of keys across a and b
        for (var key in a) {
          keys[key] = 1;
        }
        for (var key in b) {
          if (!(key in keys)) return false; // b contains a key that a does not
          keys[key]++;
        }
        var ret = true;
        for (var key in keys) {
          if (keys[key] !== 2) return false; // a contains a key that b does not
          ret = ret && _deepEquals(a[key], b[key]);
          if (!ret) return ret; // shortcut if false
        }
        return ret;
      }
      // for string, number, boolean, date, etc.
      return a === b;
    },
    sortFrequentItems: function(frequent_items, count) {
      // takes a dictionary of all frequent items
      // gives an array of the <count> most frequent items, sorted by frequency
      var values = [];
      for (var key in frequent_items) {
        values.push({
          'key': key,
          'value': frequent_items[key]
        });
      }
      values.sort(function(a,b) {
        return b.value - a.value;
      });
      return values.slice(0, count);
    },
    formatElapsed: function(ms) {
      var d = new Date(ms);
      var pieces = {
        'hours': d.getUTCHours(),
        'minutes': d.getUTCMinutes(),
        'seconds': d.getUTCSeconds()
      };
      for (var key in pieces) {
        var s = String(pieces[key]);
        if (s.length < 2) {
          s = '0' + s;
        }
        pieces[key] = s;
      }
      return pieces.hours + ':' + pieces.minutes + ':' + pieces.seconds;
    },
    numberWithSuffix: function(i) {
      // Does not work with numbers > 100
      if (i > 100) {
          return String(i);
      }
      var j = i % 10;
      if (j === 1 && i !== 11) {
          return i + "st";
      }
      if (j === 2 && i !== 12) {
          return i + "nd";
      }
      if (j === 3 && i !== 13) {
          return i + "rd";
      }
      return i + "th";
    },
    formatNumber: function(n) {
      if (n === null) {
        // Infinity, -Infinity, and NaN come in as null
        return NaN;
      }

      if ((n < 0.001 && n > 0) ||
          (n > -0.001 && n < 0)) {
        // use fixed precision
        return n.toFixed(4);
      }
      if ((n > Math.pow(2, 40)) ||
          (n < -Math.pow(2, 40))) {
        // use variable precision
        return n.toPrecision(4);
      }

      // let toLocaleString do its thing
      return n.toLocaleString();
    },
    siPrefixToShortScale: function(prefix) {
      // maps according to http://en.wikipedia.org/wiki/Metric_prefix
      var map = {
        'Y': 'septillion',
        'Z': 'sextillion',
        'E': 'quintillion',
        'P': 'quadrillion',
        'T': 'trillion',
        'G': 'billion',
        'M': 'million',
        'k': 'thousand',
        'h': 'hundred',
        'da': 'ten',
        '': null,
        'd': 'tenth',
        'c': 'hundredth',
        'm': 'thousandth',
        'µ': 'millionth',
        'n': 'billionth',
        'p': 'trillionth',
        'f': 'quadrillionth',
        'a': 'quintillionth',
        'z': 'sextillionth',
        'y': 'septillionth'
      };
      if (prefix in map) {
        return map[prefix];
      }
      Util.assert(false); // should not get here
      return null;
    },
    getCookie: function(name) {
      var r = document.cookie.match("\\b" + name + "=([^;]*)\\b");
      return r ? r[1] : undefined;
    },
    getSketchPollInterval: function(rowCount) {
      if (rowCount < pollIntervalThresholds[0][0]) {
        return pollIntervalThresholds[0][1];
      }
      if (rowCount > pollIntervalThresholds[1][0]) {
        return pollIntervalThresholds[1][1];
      }
      return pollIntervalScale(rowCount);
    }
  };
});
