import graphlab.canvas.handlers
import graphlab.canvas.server
import graphlab.canvas.state
import graphlab.connect as _mt

import codecs
import json
import os
import string
import webbrowser

class Target(object):
    """
    A base class that serves as an interface for targets.
    """
    # instance variables
    state = None
    server = None

    def __init__(self):
        self.state = graphlab.canvas.state.State()
        # TODO server is not necessary in static IPython Notebook
        self.server = graphlab.canvas.server.Server(self.state)

    # methods
    def show(self, variable):
        # track metrics on variable type
        if variable is not None:
            _mt._get_metric_tracker().track('canvas.show.%s' % type(variable).__name__)
            self.state.set_selected_variable(variable)
        else:
            _mt._get_metric_tracker().track('canvas.show')
        self.server.start()

    def add_variable(self, name, variable):
        self.state.add_variable(name, variable)

class InteractiveTarget(Target):
    """
    An interactive browser-based visualization canvas.
    """
    def show(self, variable=None):
        super(InteractiveTarget, self).show(variable)
        if not(self.server.alive()):
            webbrowser.open('http://localhost:' + str(self.server.get_port()) + "/index.html")

class IPythonTarget(Target):
    """
    Visualization rendered in an IPython Notebook output cell.
    """
    @staticmethod
    def __readFile(filename):
        """
        Read a file from the filesystem with utf-8 encoding and return the contents as a string.
        """
        dir = os.path.dirname(__file__)
        filename = os.path.join(dir, filename)
        contents = None
        with codecs.open(filename, encoding='utf-8') as f:
          contents = f.read()
        return contents

    @staticmethod
    def __makeCSS():
        """
        Generate a string of JavaScript that will put Canvas CSS on the page.
        """
        css = IPythonTarget.__readFile('canvas.css')
        js = u"""
            var s = document.createElement('style');
            s.appendChild(document.createTextNode("%s"));
            document.head.appendChild(s);
        """ % css.replace('"', '\\"').replace('\n', '')
        return js

    def __makeJS(self, data, view):
        """
        Generate a JavaScript snippet that will load Canvas in an output cell
        of IPython Notebook.
        """
        # requirejs dependencies for our code snippet
        dependencies = ['react', view]
        # dependency files (we will embed this code in the output cell)
        # order matters! these files must be listed in dependency order
        dependencyFiles = [
            'common_util.js',
            'common_controls.js',
            'logging.js',
            'state.js',
            'taskqueue.js',
            'plot_frequency.js',
            'plot_box.js',
            'plot_boxes.js',
            'plot_quantilehist.js',
            'plot_graph.js',
            'plots.js',
            'sframe.js',
            'sarray.js',
            'sgraph.js',
        ]
        dependencyFiles = string.join(map(IPythonTarget.__readFile, dependencyFiles), '\n')
        # Require paths must point to real HTTP URLs (not localhost or a file path)
        # since when we save the notebook to HTML (nbconvert) they need to still work
        # without a running instance of GraphLab Create or IPython Notebook engine.
        # In theory we could embed this code directly in the notebook (like dependencyFiles
        # above) but react.js doesn't work with this technique (if require.js is loaded
        # on the page, which it is in IPython Notebook, then react MUST be loaded through
        # requirejs, which means it has to come from a URL).
        requirePaths = json.dumps({
            "react": "http://fb.me/react-with-addons-0.10.0.min",
            "d3": "http://d3js.org/d3.v3.min",
            "dagre": "http://cpettitt.github.io/project/dagre/latest/dagre.min",
            "topojson": "http://d3js.org/topojson.v1.min",
            "vega": "http://trifacta.github.io/vega/vega.min",
            "chai": "http://chaijs.com/chai",
            "bowser": "http://www.javascriptoo.com/application/html/js/ded/bowser/bowser.min",
        })
        # Code snippet to include dependency JS files, CSS styles, and a require call to run Canvas
        js = u"""
            %(dependencyFiles)s

            %(css)s

            require.config({
                'paths': %(requirePaths)s
            });

            // figure out whether we are in IPython Notebook or HTML output
            // in the case of HTML output (nbconvert) we need to output to our own <script>
            // tag's parent element. In the case of a real IPython Notebook we
            // can use the provided "element" variable.
            var e = null;
            if (typeof element == 'undefined') {
                var scripts = document.getElementsByTagName('script');
                var thisScriptTag = scripts[scripts.length-1];
                var parentDiv = thisScriptTag.parentNode;
                e = document.createElement('div');
                parentDiv.appendChild(e);
            } else {
                e = element[0];
            }

            require(%(dependencies)s, function(React, Component) {React.renderComponent(Component.View(%(data)s), e);});
        """ % {
            "dependencies": dependencies,
            "dependencyFiles": dependencyFiles,
            "data": data,
            "css": self.__makeCSS(),
            "requirePaths": requirePaths
        }
        return js

    def show(self, variable=None):
        if variable is None:
            raise "Interactive canvas is not available in IPython mode. Please provide a selected_variable to canvas or use an interactive mode."
        super(IPythonTarget, self).show(variable)
        (name, view) = self.state.get_selected_variable()
        assert(type(name) == tuple)
        assert(isinstance(view, graphlab.canvas.views.base.BaseView))

        data = None
        selected_variable = graphlab.canvas.handlers.expand_selected_variable(self.state)
        data = {
            'selected_variable': selected_variable,
            'ipython': True,
        }
        data.update(view.get_staticdata())

        import IPython
        return IPython.core.display.Javascript(
            data=self.__makeJS(json.dumps(data), view.objectType.lower())
        )
