/** @jsx React.DOM */
// Task queue for processing application state changes
// with progress shown in the UI

define('taskqueue',
      ['common_util', 'state', 'logging'],
      function(Util, State, Logging) {

  function Task() {
    this.canceled = false;
    this.startTime = null;
    this.endTime = null;
    // callback functions w/ fluent API, as in jQuery.get
    this.fn_always = [];
    this.fn_done = [];
    this.fn_fail = [];
  }
  Task.prototype.toString = function() {
    return JSON.stringify({
      'canceled': this.canceled,
      'startTime': this.startTime,
      'endTime': this.endTime
    });
  };
  Task.prototype.always = function(fn) {
    this.fn_always.push(fn);
    return this; // return this to allow fluent api
  };
  Task.prototype.done = function(fn) {
    this.fn_done.push(fn);
    return this; // return this to allow fluent api
  };
  Task.prototype.fail = function(fn) {
    this.fn_fail.push(fn);
    return this; // return this to allow fluent api
  };
  Task.prototype.progress = function() {
    return 0; // don't know progress
  };
  Task.prototype.operation = function() {
    return ''; // don't have a specific operation
  };
  Task.prototype.running = function() {
    return this.startTime != null && this.endTime == null;
  };
  Task.prototype.finished = function() {
    return this.startTime != null && this.endTime != null;
  };
  Task.prototype.elapsed = function() {
    if (this.finished()) {
      return this.endTime - this.startTime;
    }
    else if (this.running()) {
      return Date.now() - this.startTime;
    }
    else {
      return 0; // hasn't even started yet
    }
  };
  Task.prototype.run = function() {
    Util.assert(!this.finished());
    Util.assert(!this.running());
    this.startTime = Date.now();
  };
  Task.prototype.end = function() {
    // subclasses can pass arguments to end() to pass them into the callback
    // functions

    this.endTime = Date.now();
    var applyAll = function(fnArray, args) {
      fnArray.forEach(function(fn) {fn.apply(this, args);}.bind(this));
    }.bind(this);
    if (!this.canceled) {
      applyAll(this.fn_done, arguments);
    } else {
      applyAll(this.fn_fail, arguments);
    }
    applyAll(this.fn_always, arguments);
  };
  Task.prototype.cancel = function() {
    if (!this.finished()) {
      this.canceled = true;
      this.end.apply(this, arguments);
    }
  };
  function SingleTask(url, opName) {
    Task.call(this);
    this.url = url;
    this.op = opName;
    this.request = null;
  }
  SingleTask.prototype.toString = function() {
    return JSON.stringify({
      'url': this.url,
      'op': this.op,
      'canceled': this.canceled,
      'startTime': this.startTime,
      'endTime': this.endTime
    });
  };
  SingleTask.prototype = new Task();
  SingleTask.prototype.cancel = function() {
    if (this.request !== null) {
      this.request.abort(); // stop the GET request
    }
    Task.prototype.cancel.apply(this, arguments);
  };
  SingleTask.prototype.run = function() {
    Task.prototype.run.call(this);
    this.request = $.get(this.url).done(function(data) {
      if (this.canceled) {
        return;
      }
      this.end(data);
    }.bind(this)).fail(function() {
      this.cancel.apply(this, arguments);
    }.bind(this));
  };
  SingleTask.prototype.operation = function() {
    return this.op;
  };
  function ProgressTask(url, opName, progressFn, pollInterval) {
    SingleTask.call(this, url, opName);
    this._progressFn = progressFn;
    this._progress = 0;
    this._timeout = null;
    this._pollInterval = pollInterval;
  }
  ProgressTask.prototype = new SingleTask();
  ProgressTask.prototype.run = function() {
    Task.prototype.run.call(this);
    var _this = this;
    update();
    function update() {
      $.get(_this.url).done(function(data) {
        if (_this.canceled) {
          return;
        }
        _this._progressFn(data);
        if (data.complete) {
          _this.end(data);
          _this._progress = 1;
        } else {
          // still running
          _this._progress = data.progress;
          _this._timeout = setTimeout(update, this._pollInterval);
        }
      }.bind(_this)).fail(function() {
        this.cancel.apply(this, arguments);
      }.bind(_this));
    }
  };
  ProgressTask.prototype.progress = function() {
    return this._progress;
  };
  ProgressTask.prototype.cancel = function() {
    clearTimeout(this._timeout);
    SingleTask.prototype.cancel.call(this);
  };
  function MultiTask(taskList) {
    Task.call(this);
    this.tasks = taskList;
    this.currentTask = 0;
  }
  MultiTask.prototype = new Task();
  MultiTask.prototype.progress = function() {
    var pct = this.currentTask / this.tasks.length;
    if (this.tasks[this.currentTask] instanceof ProgressTask) {
      // if the current task is a ProgressTask, include its progress in the
      // total progress calculation (for its portion of the total)
      pct += this.tasks[this.currentTask].progress() / this.tasks.length;
    }
    return pct;
  };
  MultiTask.prototype.operation = function() {
    return this.tasks.map(function(task) {
      return task.operation();
    });
  };
  MultiTask.prototype.run = function() {
    Task.prototype.run.call(this);
    var _this = this;
    function step() {
      if (_this.canceled) {
        return;
      }

      if (_this.currentTask === _this.tasks.length) {
        // done
        _this.end();
      }
      else if (_this.tasks[_this.currentTask].finished()) {
        // finished a subtask, move on to the next
        _this.currentTask++;
      }
      else if (!_this.tasks[_this.currentTask].running()) {
        // current subtask hasn't started yet, kick it off
        _this.tasks[_this.currentTask].run();
      }

      // falls through, unless we are finished check again soon on subtasks
      if (!_this.finished()) {
        setTimeout(step, 1000);
      }
    }
    step();
  };
  MultiTask.prototype.cancel = function() {
    for (var i=0; i<this.tasks.length; i++) {
      this.tasks[i].cancel();
    }
    Task.prototype.cancel.call(this);
  };

  var previousProgress = 0;
  var previousFinished = false;
  function processTasks() {
    if (g_tasks.length === 0) {
      // nothing to do, wait
      return;
    }

    if (g_tasks[0].running()) {
      // nothing to do, wait
    } else if (g_tasks[0].finished()) {
      if (g_tasks.length > 1) {
        // remove completed task and wait for the next interval
        Logging.log('removing completed task');
        var removed = g_tasks.shift();
        Logging.log(removed.toString());
      }
    } else {
      // current task hasn't started yet, kick it off
      Logging.log('starting new task');
      Logging.log(g_tasks[0].toString());
      g_tasks[0].run();
    }

    var currentTask = g_tasks[0];
    // avoid calling setState if possible to avoid React
    // re-rendering or state comparison
    var progress = currentTask.progress();
    var finished = currentTask.finished();
    if (progress !== previousProgress ||
        finished !== previousFinished) {
      previousProgress = progress;
      previousFinished = finished;
      State.set({
        'task': {
          'progress': progress,
          'finished': finished
        }
      });
    }
  }
  
  function cancelAllTasks() {
    // cancel all the existing tasks (we are switching variables or something)
    for (var i=0; i<g_tasks.length; i++) {
      g_tasks[i].cancel();
    }
  }

  var g_tasks = [];
  return {
    'add': function(task) {g_tasks.push(task);},
    'cancelAll': cancelAllTasks,
    'start': function() { setInterval(processTasks, 200); },
    'Task': Task,
    'SingleTask': SingleTask,
    'ProgressTask': ProgressTask,
    'MultiTask': MultiTask
  };
});
