"""
The graph analytics toolkit contains methods for analyzing a
:class:`~graphlab.SGraph`. Each method takes an input graph
and returns a model object, which contains the run time, an
:class:`~graphlab.SFrame` with the desired output for each
vertex, and a new graph whose vertices contain the output as an attribute. Note
that all of the methods in the graph analytics toolkit are available from the
top level graphlab import.

For this example we download a dataset of James Bond characters to an
SFrame, then build the SGraph.

.. sourcecode:: python

    >>> from graphlab import SFrame, SGraph

    >>> data = graphlab.SFrame.read_csv('http://s3.amazonaws.com/GraphLab-Datasets/bond/bond_edges.csv')
    >>> g = SGraph().add_edges(data, src_field='src', dst_field='dst')

The :py:func:`connected_components.create()
<graphlab.connected_components.create>` method finds all weakly connected
components in the graph. A connected component is a group of vertices such that
there is a path between each vertex in the component and all other vertices in
the group. If two vertices are in different connected components there is no
path between them.

.. sourcecode:: python

    >>> from graphlab import connected_components
    >>> cc = connected_components.create(g)
    >>> print cc.summary()
    >>> print cc.list_fields()

    >>> cc_ids = cc.get('componentid')  # an SFrame
    >>> cc_ids = cc['componentid']      # equivalent to the above line
    >>> cc_graph = cc['graph']  # a GraphLab SGraph

The :py:func:`shortest_path.create() <graphlab.connected_components.create>`
method finds the shortest directed path distance from a single source vertex to
all other vertices. The output model contains this information and a method to
retrieve the actual path to a particular vertex.

.. sourcecode:: python

    >>> from graphlab import shortest_path
    >>> sp = shortest_path.create(g, source_vid=123)
    >>> sp_sframe = sp['distance']
    >>> sp_graph = sp['graph']
    >>> path = sp.get_path('98')

A graph triangle is a complete subgraph of three vertices. The number of
triangles to which a vertex belongs is an indication of the connectivity of the
graph around that vertex.

.. sourcecode:: python

    >>> from graphlab import triangle_counting
    >>> tc = triangle_counting.create(g)
    >>> tc_out = tc['triangle_count']

The :py:func:`pagerank.create() <graphlab.pagerank.create>` method indicates the
centrality of each node in the graph.

.. sourcecode:: python

    >>> from graphlab import pagerank
    >>> pr = pagerank.create(g)
    >>> pr_out = pr['pagerank']

The K-core decomposition recursively removes vertices from the graph with degree
less than k. The value of k where a vertex is removed is its core ID; the
:py:func:`kcore.create() <graphlab.kcore.create>` method returns the core ID for
all vertices in the graph.

.. sourcecode:: python

    >>> from graphlab import kcore
    >>> kc = kcore.create(g)
    >>> kcore_id = kc['coreid']

Graph coloring assigns each vertex in the graph to a group in such a way that no
two adjacent vertices share the same group.

.. sourcecode:: python

    >>> from graphlab import graph_coloring
    >>> color = graph_coloring.create(g)
    >>> color_id = color['colorid']
    >>> num_colors = color['num_colors']

For more information on the models in the graph analytics toolkit, please see
the tutorial at http://www.graphlab.com/products/create/tutorials.html.

"""

import pagerank
import triangle_counting
import connected_components
import kcore
import graph_coloring
import shortest_path
