r"""
The GraphLab regression toolkit includes linear and logistic regression models
that can be trained to make predictions about new data.

GraphLab regression models accept training and prediction data in the form of an
:class:`~graphlab.SFrame`. Training datasets should contain a column for the
target variable and one or more columns representing feature variables. For
predictions on new data, the SFrame need not contain the target column.

The :py:class:`~graphlab.linear_regression.LinearRegressionModel` model
predicts a scalar target variable as a linear function of one or more feature
variables. The :py:class:`~graphlab.linear_regression.LinearRegressionModel`
also support L1 and L2 regularized formulations.

.. sourcecode:: python

    # Load the data 
    >>> import graphlab as gl
    >>> data =  gl.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')
    
    # Make a linear regression model
    >>> model = gl.linear_regression.create(data, target='price', features=['bath', 'bedroom', 'size'])

    # Here, all columns (other than the target) are used as features.
    >>> model = gl.linear_regression.create(data, target='price')

    # Extract the coefficients as an SFrame
    >>> coefficients = model['coefficients']
    
    # Make predictions as an SArray
    >>> predictions = model.predict(data)   
    
    # Evaluate the model 
    >>> results = m.evaluate(data)          


The :class:`~graphlab.logistic_regression.LogisticRegressionModel` uses a binary
target variable :math:`y` instead of a scalar. For each observation, the
probability that :math:`y=1` (instead of 0) is modeled as the logistic
function of a linear combination of the feature values. Logistic regression
predictions may take the form of the linear combination (also known as the
margin), the probability that :math:`y = 1`, or a binary class value, which is
obtained by thresholding the probability at 0.5. The `~graphlab.evaluation`
toolkit gives you more flexible evaluation metrics. As is the case with
the linear regression module, the logistic regression module also supports
L1 and L2 regularized formulations.

.. sourcecode:: python


    # Load the data 
    >>> import graphlab as gl
    >>> data =  gl.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

    # Make sure the target is binary 0/1
    >>> data['expensive'] = data['price'] > 30000
    
    # Make a logistic regression model
    >>> model = gl.logistic_regression.create(data, target='expensive', features=['bath', 'bedroom', 'size'])

    # Here, all columns (other than the target) are used as features.
    >>> model = gl.logistic_regression.create(data, target='expensive')

    # Extract the coefficients as an SFrame
    >>> coefficients = model['coefficients'] 
    
    # Make predictions (as margins, probability, or class) as an SArray
    >>> predictions = model.predict(data)    
    >>> predictions = model.predict(data, output_type='probability')
    >>> predictions = model.predict(data, output_type='margin')       

    # Evaluate the model 
    >>> results = m.evaluate(data)               # a dictionary

"""

from graphlab.toolkits.regression import linear_regression
from graphlab.toolkits.regression import logistic_regression
